/*
 * Copyright 2004-2006 H2 Group. Licensed under the H2 License, Version 1.0 (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package org.h2.command.ddl;

import java.sql.SQLException;

import org.h2.command.Parser;
import org.h2.engine.Comment;
import org.h2.engine.Constants;
import org.h2.engine.Database;
import org.h2.engine.DbObject;
import org.h2.engine.Session;
import org.h2.expression.Expression;
import org.h2.message.Message;
import org.h2.schema.Schema;
import org.h2.table.Table;

public class SetComment extends DefineCommand {

    private String schemaName;
    private String objectName;
    private boolean column;
    private String columnName;
    private int objectType;
    private Expression expr;

    public SetComment(Session session) {
        super(session);
    }

    public int update() throws SQLException {
        session.commit();
        Database db = session.getDatabase();
        session.getUser().checkAdmin();
        DbObject object = null;
        int errorCode = Message.GENERAL_ERROR_1;
        if(schemaName == null) {
            schemaName = Constants.SCHEMA_MAIN;
        }
        Schema schema = db.findSchema(schemaName);
        switch(objectType) {
        case DbObject.CONSTANT:
            object = schema.getConstant(session, objectName);
            break;
        case DbObject.CONSTRAINT:
            object = schema.getConstraint(objectName);
            break;
        case DbObject.FUNCTION_ALIAS:
            object = db.findFunctionAlias(objectName);
            errorCode = Message.FUNCTION_ALIAS_NOT_FOUND_1;
            break;
        case DbObject.INDEX:
            object = schema.getIndex(objectName);
            break;
        case DbObject.ROLE:
            object = db.findRole(objectName);
            errorCode = Message.ROLE_NOT_FOUND_1;
            break;
        case DbObject.SCHEMA:
            object = db.findSchema(objectName);
            errorCode = Message.SCHEMA_NOT_FOUND_1;
            break;
        case DbObject.SEQUENCE:
            object = schema.getSequence(objectName);
            break;
        case DbObject.TABLE_OR_VIEW:
            object = schema.getTableOrView(session, objectName);
            break;
        case DbObject.TRIGGER:
            object = schema.findTrigger(objectName);
            errorCode = Message.TRIGGER_NOT_FOUND_1;
            break;
        case DbObject.USER:
            object = db.getUser(objectName);
            break;
        case DbObject.USER_DATATYPE:
            object = db.findUserDataType(objectName);
            errorCode = Message.USER_DATA_TYPE_ALREADY_EXISTS_1;
            break;
        }
        if(object == null) {
            throw Message.getSQLException(errorCode, objectName);
        }
        String fullName = Parser.quoteIdentifier(schemaName);
        fullName += "." + Parser.quoteIdentifier(objectName);
        if(column) {
            fullName += "." + Parser.quoteIdentifier(columnName);
        }
        String text = expr.getValue(session).getString();
        String key = Comment.getKey(objectType, column, fullName);
        Comment comment = db.findComment(key);
        if(column) {
            Table table = (Table) object;
            table.getColumn(columnName).setComment(text);
        } else {
            object.setComment(text);
        }
        if(comment == null) {
            if(text == null) {
                // reset a non-existing comment - nothing to do
            } else {
                int id = getObjectId(false, false);
                comment = new Comment(db, id, objectType, column, fullName);
                comment.setCommentText(text);
                db.addDatabaseObject(session, comment);
            }
        } else {
            if(text == null) {
                db.removeDatabaseObject(session, comment);
            } else {
                comment.setCommentText(text);
                db.update(session, comment);
            }
        }
        return 0;
    }

    public void setCommentExpression(Expression expr) {
        this.expr = expr;
    }

    public void setObjectName(String objectName) {
        this.objectName = objectName;
    }

    public void setObjectType(int objectType) {
        this.objectType = objectType;
    }

    public void setColumnName(String columnName) {
        this.columnName = columnName;
    }

    public void setSchemaName(String schemaName) {
        this.schemaName = schemaName;
    }

    public void setColumn(boolean column) {
        this.column = column;
    }

}
