/*
 * Copyright 2006-2011 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.codegen.maven;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.URL;
import java.net.URLClassLoader;
import java.sql.Connection;
import java.sql.Driver;
import java.sql.SQLException;
import java.util.Map;
import java.util.Properties;

import javax.sql.DataSource;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.resolver.ArtifactNotFoundException;
import org.apache.maven.artifact.resolver.ArtifactResolutionException;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.seasar.codegen.CodeGenConfig;
import org.seasar.codegen.DDL;
import org.seasar.codegen.ImportCodeData;
import org.seasar.codegen.Java;
import org.seasar.codegen.OutputCode;
import org.seasar.codegen.Resource;
import org.seasar.codegen.ant.Directory;
import org.seasar.codegen.ant.DirectoryList;
import org.seasar.codegen.ant.EachGen;
import org.seasar.codegen.convert.NameConverter;
import org.seasar.codegen.dbms.Dbms;
import org.seasar.codegen.impl.CodeGenConfigImpl;
import org.seasar.codegen.impl.DDLOutputCodeImpl;
import org.seasar.codegen.impl.DatabaseImportCodeData;
import org.seasar.framework.container.ComponentDef;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.factory.S2ContainerFactory;
import org.seasar.framework.container.impl.ComponentDefImpl;
import org.seasar.framework.exception.ClassNotFoundRuntimeException;
import org.seasar.framework.exception.IORuntimeException;
import org.seasar.framework.exception.IllegalAccessRuntimeException;
import org.seasar.framework.exception.InstantiationRuntimeException;
import org.seasar.framework.exception.SRuntimeException;
import org.seasar.framework.util.ClassUtil;

/**
 * Goal which touches a timestamp file.
 * 
 * @goal generate
 * 
 * @phase generate-sources
 */
public class CodegenMojo extends AbstractMojo {

    /**
     * Javaソースの出力先です。
     * 
     * @parameter default-value="src/main/java"
     * 
     */
    private String javaDir;

    /**
     * リソースファイルの出力先です。
     * 
     * @parameter default-value="src/main/resources"
     */
    private String resourceDir;

    /**
     * DDLの出力先です。
     * 
     * @parameter default-value="src/main/ddl"
     */
    private String ddlDir;

    /**
     * ヘッダーテンプレートのパスです。
     * 
     * @parameter
     */
    private String headerTemplatePath;

    /**
     * 排他制御用のバージョン番号のプロパティ名です。
     * 
     * @parameter
     */
    private String versionNoPropertyName;

    /**
     * 排他制御用のタイムスタンプのプロパティ名です。
     * 
     * @parameter
     */
    private String timestampPropertyName;

    /**
     * 主キーの生成方法を指定します。
     * 
     * @parameter
     */
    private String identityType;

    /**
     * alldao.diconを出力するかを指定します。
     * 
     * @parameter default-value="false"
     */
    private boolean outputAllDaoDicon;

    /**
     * DDLを出力するかを指定します。
     * 
     * @parameter default-value="false"
     */
    private boolean outputDDL;

    /**
     * 出力エンコーディングを指定します。
     * 
     * @parameter default-value="UTF-8"
     */
    private String encoding;

    /**
     * 入力ファイルのパスを指定します。
     * 
     * @parameter
     * @required
     */
    private String inputFile;

    /**
     * 出力パッケージを指定します。
     * 
     * @parameter
     * @required
     */
    private String packageName;

    /**
     * 入力処理クラスを指定します。
     * 
     * @parameter default-value="org.seasar.codegen.impl.JiemamyImportCodeData"
     */
    private String importCodeDataClass;

    /**
     * DBクラスを指定します。
     * 
     * @parameter
     * @required
     */
    private String dbClass;

    /**
     * 外部キーのプロパティの変換処理クラスを指定します。
     * 
     * @parameter default-value=
     *            "org.seasar.codegen.convert.impl.FKNameChildNameConverterImpl"
     */
    private String fkNameConverterClass;

    /**
     * 出力クラスのプロパティ・クラス名の変換処理クラスを指定します。
     * 
     * @parameter 
     *            default-value="org.seasar.codegen.convert.impl.CamelConverterImpl"
     */
    private String nameConverterClass;

    /**
     * DBスキーマから読み込む時の対象スキーマです。
     * 
     * @parameter
     */
    private String schemeName;

    /**
     * DBスキーマから読み込む時の対象テーブルの配列です。
     * 
     * @parameter
     */
    private String[] tableNames;

    /**
     * DBスキーマから読み込む時の対象外テーブルの正規表現です。
     * 
     * @parameter
     */
    private String ignoreTablePattern;

    /** @parameter default-value="${localRepository}" */
    private org.apache.maven.artifact.repository.ArtifactRepository localRepository;

    /** @parameter default-value="${project.remoteArtifactRepositories}" */
    @SuppressWarnings({ "rawtypes" })
    private java.util.List remoteRepositories;

    /**
     * スキーマから生成する場合のJDBCドライバのgroupIdです。
     * 
     * @parameter
     **/
    private String driverGroupId;

    /**
     * スキーマから生成する場合のJDBCドライバのartifactIdです。
     * 
     * @parameter
     */
    private String driverArtifactId;

    /**
     * スキーマから生成する場合のJDBCドライバのversionです。
     * 
     * @parameter
     * 
     */
    private String driverVersion;

    /**
     * スキーマから生成する場合のドライバのクラス名です。
     * 
     * @parameter
     */
    private String driverClassName;

    /**
     * スキーマから生成する場合の接続先URLです。
     * 
     * @parameter
     */
    private String url;

    /**
     * スキーマから生成する場合の接続先ユーザ名です。
     * 
     * @parameter
     */
    private String userName;

    /**
     * スキーマから生成する場合の接続先パスワードです。
     * 
     * @parameter
     */
    private String password = "";

    /**
     * JiemamyおよびDBスキーマから生成する場合の主キーとシーケンス名からのマッピングです。
     * 
     * @parameter
     */
    private Map<String, String> sequenceMapping;

    /** @component */
    private org.apache.maven.artifact.factory.ArtifactFactory artifactFactory;

    /** @component */
    private org.apache.maven.artifact.resolver.ArtifactResolver resolver;

    public void execute() throws MojoExecutionException, MojoFailureException {

        DirectoryList directoryList = new DirectoryList();
        directoryList.add("java", javaDir);
        directoryList.add("resource", resourceDir);
        directoryList.add("createtable", ddlDir);

        CodeGenConfigImpl configImpl = new CodeGenConfigImpl();
        configImpl.setHeaderTemplatePath(headerTemplatePath);
        configImpl.setVersionNoPropertyName(versionNoPropertyName);
        configImpl.setTimestampPropertyName(timestampPropertyName);
        configImpl.setIdentityType(identityType);
        configImpl.setOutputAllDaoDicon(outputAllDaoDicon);
        configImpl.setOutputDDL(outputDDL);
        configImpl.setEncoding(encoding);
        configImpl.setInputFile(inputFile);
        configImpl.setPackageName(packageName);
        System.out.println(sequenceMapping);
        if (sequenceMapping != null) {
            configImpl.setSequnceMapping(sequenceMapping);
        }

        ClassConfig classConfig = new ClassConfig(importCodeDataClass, dbClass,
                fkNameConverterClass, nameConverterClass);
        DataSource dataSource = null;
        if (driverClassName != null) {
            try {
                Artifact artifact = artifactFactory.createArtifact(
                        driverGroupId, driverArtifactId, driverVersion,
                        "compile", "jar");

                resolver.resolve(artifact, remoteRepositories, localRepository);
                dataSource = new CodegenMojo.DataSourceImpl(driverClassName,
                        url, userName, password, artifact.getFile());
            } catch (ArtifactResolutionException e) {
                throw new MojoExecutionException(e.getMessage(), e);
            } catch (ArtifactNotFoundException e) {
                throw new MojoExecutionException(e.getMessage(), e);
            }
        }
        DatabaseConfig databaseConfig = new DatabaseConfig(schemeName,
                tableNames, ignoreTablePattern, dataSource);

        doExecute(directoryList, configImpl, classConfig, directoryList,
                databaseConfig);
    }

    void doExecute(DirectoryList list, CodeGenConfig codeGenConfig,
            ClassConfig classConfig, DirectoryList directoryList,
            DatabaseConfig databaseConfig) {
        S2Container s2 = S2ContainerFactory.create("app_gen.dicon");
        doGenarate(codeGenConfig, classConfig, directoryList, databaseConfig,
                s2);

    }

    private void doGenarate(CodeGenConfig codeGenConfig,
            ClassConfig classConfig, DirectoryList directory,
            DatabaseConfig databaseConfig, S2Container s2) {
        S2Container container = S2ContainerFactory.create();
        try {
            ComponentDef def = new ComponentDefImpl(
                    ClassUtil.forName(classConfig.getImportCodeDataClass()));
            container.register(def);
            ComponentDef defDbms = new ComponentDefImpl(
                    ClassUtil.forName(classConfig.getDbClass()));
            container.register(defDbms);
            ComponentDef defFkName = new ComponentDefImpl(
                    ClassUtil.forName(classConfig.getFkNameConverterClass()));
            container.register(defFkName);
            ComponentDef defNameConverter = new ComponentDefImpl(
                    ClassUtil.forName(classConfig.getNameConverterClass()));
            container.register(defNameConverter);
            container.register(codeGenConfig);
            if (databaseConfig.isUse()) {
                container.register(databaseConfig.getDataSource());
            }

            S2Container child = S2ContainerFactory.create("CodeGen.dicon");
            child.init();
            container.include(child);
            container.init();
            s2.include(container);
            s2.init();
            ComponentDef[] components = s2
                    .findAllComponentDefs(OutputCode.class);
            injectDatabaseProperty(s2.getComponent(ImportCodeData.class),
                    databaseConfig);
            for (ComponentDef componentDef : components) {
                OutputCode outputCode = (OutputCode) componentDef
                        .getComponent();
                outputCode.setCodegenConfig(codeGenConfig);
                outputCode.setNameConverter((NameConverter) container
                        .getComponent(NameConverter.class));
                if (outputCode.getClass().isAnnotationPresent(Java.class)) {
                    doGenarate(codeGenConfig, classConfig, outputCode,
                            directory.getDirectory("java"), s2);
                } else if (outputCode.getClass().isAnnotationPresent(
                        Resource.class)) {
                    doGenarate(codeGenConfig, classConfig, outputCode,
                            directory.getDirectory("resource"), s2);
                } else if (outputCode.getClass().isAnnotationPresent(DDL.class)) {
                    ((DDLOutputCodeImpl) outputCode).setDbms((Dbms) container
                            .getComponent(Dbms.class));
                    doGenarate(codeGenConfig, classConfig, outputCode,
                            directory.getDirectory("createtable"), s2);
                } else {
                    throw new SRuntimeException("ECDG0003",
                            new Object[] { outputCode.getClass() });
                }
            }

        } finally {
            container.destroy();

        }

    }

    private void injectDatabaseProperty(Object component,
            DatabaseConfig databaseConfig) {
        if (component instanceof DatabaseImportCodeData) {
            DatabaseImportCodeData importCodeData = (DatabaseImportCodeData) component;
            importCodeData.setSchemaName(databaseConfig.getSchemeName());
            importCodeData.setIgnoreTablePattern(databaseConfig
                    .getIgnoreTablePattern());
            for (String table : databaseConfig.getTableNames()) {
                importCodeData.addTable(table);
            }
        }

    }

    private void doGenarate(CodeGenConfig codeGenConfig,
            ClassConfig classConfig, OutputCode outputCode,
            Directory directory, S2Container s2) {
        EachGen eachGen = new EachGen();
        eachGen.setContainer(s2);
        eachGen.setCsvFile(new File(codeGenConfig.getInputFile()));
        eachGen.setDestinationDir(new File(directory.getDestination()));
        eachGen.setOutputCode(outputCode);
        eachGen.generate();
    }

    static class ClassConfig {
        private String importCodeDataClass;

        private String dbClass;

        private String fkNameConverterClass;

        private String nameConverterClass;

        public String getImportCodeDataClass() {
            return importCodeDataClass;
        }

        public String getDbClass() {
            return dbClass;
        }

        public String getFkNameConverterClass() {
            return fkNameConverterClass;
        }

        public String getNameConverterClass() {
            return nameConverterClass;
        }

        public ClassConfig(String importCodeDataClass, String dbClass,
                String fkNameConverterClass, String nameConverterClass) {
            super();
            this.importCodeDataClass = importCodeDataClass;
            this.dbClass = dbClass;
            this.fkNameConverterClass = fkNameConverterClass;
            this.nameConverterClass = nameConverterClass;
        }

    }

    static class DatabaseConfig {
        private String schemeName;
        private String[] tableNames;
        private String ignoreTablePattern;
        private DataSource dataSource;

        public DatabaseConfig(String schemeName, String[] tableNames,
                String ignoreTablePattern, DataSource dataSource) {
            super();
            this.schemeName = schemeName;
            if (tableNames == null) {
                this.tableNames = new String[0];
            } else {
                this.tableNames = tableNames;
            }
            this.ignoreTablePattern = ignoreTablePattern;
            this.dataSource = dataSource;
        }

        public String getSchemeName() {
            return schemeName;
        }

        public String[] getTableNames() {
            return tableNames;
        }

        public String getIgnoreTablePattern() {
            return ignoreTablePattern;
        }

        public DataSource getDataSource() {
            return dataSource;
        }

        public boolean isUse() {
            return dataSource != null;
        }
    }

    private static class DataSourceImpl implements DataSource {

        private String url;

        private String user;

        private String password;

        private String className;

        private File classPath;

        public DataSourceImpl(String className, String url, String user,
                String password, File driverClassPath) {
            this.className = className;
            this.url = url;
            this.user = user;
            this.password = password;
            this.classPath = driverClassPath;
        }

        public Connection getConnection() throws SQLException {
            return getConnection(user, password);
        }

        public Connection getConnection(String username, String password)
                throws SQLException {
            Properties info = new Properties();
            info.setProperty("user", username);
            info.setProperty("password", password);

            return getDriver(className, classPath).connect(this.url, info);
        }

        protected Driver getDriver(String className2, File classPath2) {
            Class<?> clazz = null;
            try {
                URL[] urls = new URL[1];
                ;
                urls[0] = classPath2.toURL();
                ClassLoader loader = new URLClassLoader(urls);
                clazz = loader.loadClass(className2);
                return (Driver) clazz.newInstance();
            } catch (IOException e) {
                throw new IORuntimeException(e);
            } catch (InstantiationException e) {
                throw new InstantiationRuntimeException(clazz, e);
            } catch (IllegalAccessException e) {
                throw new IllegalAccessRuntimeException(clazz, e);
            } catch (ClassNotFoundException e) {
                throw new ClassNotFoundRuntimeException(e);
            }

        }

        public PrintWriter getLogWriter() throws SQLException {
            return new PrintWriter(System.out);
        }

        public int getLoginTimeout() throws SQLException {
            return 0;
        }

        public void setLogWriter(PrintWriter out) throws SQLException {

        }

        public void setLoginTimeout(int seconds) throws SQLException {

        }
    }
}
