/*
 * Copyright 2004-2010 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.doma.jdbc.id;

import java.sql.Statement;

import org.seasar.doma.GenerationType;
import org.seasar.doma.jdbc.JdbcException;

/**
 * 識別子のジェネレータです。
 * <p>
 * {@link #generatePreInsert(IdGenerationConfig)} と
 * {@link #generatePostInsert(IdGenerationConfig, Statement)} のどちらか片方が
 * {@code null} ではない値を返さなければいけません。
 * <p>
 * このインタフェースの実装は、引数なしの {@code public} なコンストラクタを持たなければいけません。
 * <p>
 * このインタフェースの実装はスレッドセーフでなければいけません。
 * 
 * @author taedium
 * 
 */
public interface IdGenerator {

    /**
     * バッチ処理をサポートしているかどうかを返します。
     * 
     * @param config
     *            識別子生成の設定
     * @return サポートしている場合 {@code true}
     */
    boolean supportsBatch(IdGenerationConfig config);

    /**
     * {@link Statement#getGeneratedKeys()} をサポートしているかどうかを返します。
     * 
     * @param config
     *            識別子生成の設定
     * @return サポートしている場合 {@code true}
     */
    boolean supportsAutoGeneratedKeys(IdGenerationConfig config);

    /**
     * INSERT文にIDENTITYカラムを含めるべきかどうかを返します。
     * 
     * @param config
     *            識別子生成の設定
     * @return サポートしている場合 {@code true}
     */
    boolean includesIdentityColumn(IdGenerationConfig config);

    /**
     * INSERTの実行前に識別子を生成します。
     * 
     * @param config
     *            識別子生成の設定
     * @return 生成された識別子、サポートしていない場合 {@code null}
     * @throws JdbcException
     *             識別子の生成に失敗した場合
     */
    Long generatePreInsert(IdGenerationConfig config);

    /**
     * INSERTの実行後に識別子を生成します。
     * 
     * @param config
     *            識別子生成の設定
     * @param statement
     *            INSERT文を実行した文
     * @return 生成された識別子、サポートしていない場合 {@code null}
     * @throws JdbcException
     *             識別子の生成に失敗した場合
     */
    Long generatePostInsert(IdGenerationConfig config, Statement statement);

    /**
     * 識別子を生成する方法を返します。
     * 
     * @return 識別子を生成する方法
     */
    GenerationType getGenerationType();
}
