/*
 * Copyright 2006-2011 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.codegen.lib.impl;

import java.util.List;

import org.seasar.codegen.lib.Column;
import org.seasar.codegen.lib.ConditionElement;
import org.seasar.codegen.lib.ConditionResult;
import org.seasar.codegen.lib.Operator;
import org.seasar.codegen.lib.internal.impl.ValueValidator;

/**
 * @author azusa
 * 
 */
public class ConditionElementImpl<T> implements ConditionElement<T> {

    protected ConditionResult conditionResult;

    protected Column column;

    private final Object[] EMPTY = new Object[0];

    /**
     * コンストラクタです。
     * 
     * @param conditionResult
     *            検索条件の組み立て結果
     * @param column
     *            カラム
     */
    public ConditionElementImpl(ConditionResult conditionResult, Column column) {
        this.conditionResult = conditionResult;
        this.column = column;
    }

    public void addOrderAsc() {
        conditionResult.addOrder(column, true);
    }

    public void addOrderDesc() {
        conditionResult.addOrder(column, false);

    }

    public void equal(T value) {
        ValueValidator.validate(value, column);
        conditionResult.addBindValue(new BindValueImpl(column, Operator.EQUAL, new Object[] { value }));

    }

    public void in(T... values) {
        ValueValidator.validate(values, column);
        conditionResult.addBindValue(new BindValueImpl(column, Operator.IN, values));

    }

    public void in(List<T> values) {
        ValueValidator.validate(values, column);
        conditionResult.addBindValue(new BindValueImpl(column, Operator.IN, values.toArray()));

    }

    public void isNotNull() {
        conditionResult.addBindValue(new BindValueImpl(column, Operator.NOT_NULL, EMPTY));

    }

    public void isNull() {
        conditionResult.addBindValue(new BindValueImpl(column, Operator.NULL, EMPTY));
    }

    public void less(T value) {
        ValueValidator.validate(value, column);
        conditionResult.addBindValue(new BindValueImpl(column, Operator.LESS, new Object[] { value }));
    }

    public void lessThan(T value) {
        ValueValidator.validate(value, column);
        conditionResult.addBindValue(new BindValueImpl(column, Operator.LESSTHAN, new Object[] { value }));
    }

    public void more(T value) {
        ValueValidator.validate(value, column);
        conditionResult.addBindValue(new BindValueImpl(column, Operator.MORE, new Object[] { value }));
    }

    public void moreThan(T value) {
        ValueValidator.validate(value, column);
        conditionResult.addBindValue(new BindValueImpl(column, Operator.MORETHAN, new Object[] { value }));

    }

    public void not(T value) {
        ValueValidator.validate(value, column);
        conditionResult.addBindValue(new BindValueImpl(column, Operator.NOT, new Object[] { value }));
    }

    public void notIn(T... values) {
        ValueValidator.validate(values, column);
        conditionResult.addBindValue(new BindValueImpl(column, Operator.NOTIN, values));
    }

    public void notIn(List<T> values) {
        ValueValidator.validate(values, column);
        conditionResult.addBindValue(new BindValueImpl(column, Operator.NOTIN, values.toArray()));
    }

}
