/*
 * Copyright 2004-2006 H2 Group. Licensed under the H2 License, Version 1.0 (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package org.h2.index;

import java.sql.SQLException;

import org.h2.command.dml.Query;
import org.h2.engine.Session;
import org.h2.expression.Comparison;
import org.h2.expression.ExpressionVisitor;
import org.h2.expression.Parameter;
import org.h2.expression.ValueExpression;
import org.h2.message.Message;
import org.h2.result.LocalResult;
import org.h2.result.Row;
import org.h2.table.Column;
import org.h2.table.PlanItem;
import org.h2.table.TableView;
import org.h2.util.IntArray;
import org.h2.util.ObjectArray;
import org.h2.value.Value;

public class ViewIndex extends Index {

    private String querySQL;
    private ObjectArray originalParameters;
    private Parameter[] params;
    
    private Value[] lastValues;
    private LocalResult lastResult;

    public ViewIndex(TableView view, String querySQL, ObjectArray originalParameters) {
        super(view, 0, null, null, IndexType.createNonUnique(false));
        this.querySQL = querySQL;
        this.originalParameters = originalParameters;
    }
    
    public String getPlanSQL() {
        return querySQL;
    }

    public void close(Session session) throws SQLException {
    }

    public void add(Session session, Row row) throws SQLException {
        throw Message.getUnsupportedException();
    }

    public void remove(Session session, Row row) throws SQLException {
        throw Message.getUnsupportedException();
    }

    private int getComparisonType(int mask) {
        if((mask & IndexCondition.EQUALITY) == IndexCondition.EQUALITY) {
            return Comparison.EQUAL;
        } else if ((mask & IndexCondition.RANGE) == IndexCondition.RANGE) {
            // TODO view: how to do range queries?
            return Comparison.BIGGER_EQUAL;
        } else if ((mask & IndexCondition.START) == IndexCondition.START) {
            return Comparison.BIGGER_EQUAL;
        } else if ((mask & IndexCondition.END) == IndexCondition.END) {
            return Comparison.SMALLER_EQUAL;
        }
        throw Message.internal("unsupported mask "+mask);
    }
    
    public Query getQuery(Session session, int[] masks) throws SQLException {
        Query query = (Query)session.prepare(querySQL, true);
        IntArray paramIndex = new IntArray();
        if(masks == null) {
            double cost = query.getCost();
            PlanItem item = new PlanItem();
            item.cost = cost;
            columns = new Column[0];
            params = new Parameter[0];
            return query;
        }
        for(int i=0; i<masks.length; i++) {
            int mask = masks[i];
            if(mask == 0) {
                continue;
            }
            paramIndex.add(i);
        }
        int len = paramIndex.size();
        columns = new Column[len];
        params = new Parameter[len];
        for(int i=0; i<len; i++) {
            int idx = paramIndex.get(i);
            Column col = table.getColumn(idx);
            columns[i] = col;
            Parameter param = new Parameter(0);
            params[i] = param;
            int mask = masks[idx];
            int comparisonType = getComparisonType(mask);
            query.addGlobalCondition(param, idx, comparisonType);
        }
        String sql = query.getSQL();
        query = (Query)session.prepare(sql);
        return query;
    }

    public Cursor find(Session session, Row first, Row last) throws SQLException {
        Query query = (Query)session.prepare(querySQL, true);
        ObjectArray params = query.getParameters();
        for(int i=0; first != null && i<first.getColumnCount(); i++) {
            Value v = first.getValue(i);
            if(v != null) {
                query.addGlobalCondition(ValueExpression.get(v), i, Comparison.BIGGER_EQUAL);
            }
        }
        for(int i=0; last != null && i<last.getColumnCount(); i++) {
            Value v = last.getValue(i);
            if(v != null) {
                query.addGlobalCondition(ValueExpression.get(v), i, Comparison.SMALLER_EQUAL);
            }
        }
        boolean sameAsLast;
        if(query.isEverything(ExpressionVisitor.DETERMINISTIC)) {
            sameAsLast = lastResult != null && first == null && last == null;
        } else {
            sameAsLast = false;
        }
        for(int i=0; originalParameters != null && i<originalParameters.size(); i++) {
            Parameter orig = (Parameter) originalParameters.get(i);
            Parameter param = (Parameter) params.get(i);
            Value value = orig.getValue(session);
            if(sameAsLast) {
                if(!session.getDatabase().areEqual(value, lastValues[i])) {
                    sameAsLast = false;
                }
            }
            if(lastValues == null) {
                lastValues = new Value[originalParameters.size()];
            }            
            lastValues[i] = value;
            param.setValue(value);
        }
        LocalResult result;
        if(sameAsLast) {
            result = lastResult;
            result.reset();
        } else {
            query.setSession(session);
            result = query.query(0);
            lastResult = result;
        }
        return new ViewCursor(table, result);
    }
    
    public int getCost(int[] masks) throws SQLException {
        if(masks != null) {
            throw Message.getUnsupportedException();
        }
        return Integer.MAX_VALUE;
    }

    public void remove(Session session) throws SQLException {
        throw Message.getUnsupportedException();
    }

    public void truncate(Session session) throws SQLException {
        throw Message.getUnsupportedException();
    }

    public void checkRename() throws SQLException {
        throw Message.getUnsupportedException();
    }

    public boolean needRebuild() {
        return false;
    }
    
    public boolean canGetFirstOrLast(boolean first) {
        return false;
    }

    public Value findFirstOrLast(Session session, boolean first) throws SQLException {
        throw Message.getUnsupportedException();
    }     

}
