/*
 * Copyright 2006-2011 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.codegen.dbms;

import java.util.HashMap;
import java.util.Map;

import org.seasar.codegen.LanguageDataTypeSelectHelper;
import org.seasar.codegen.element.DataType;
import org.seasar.codegen.element.FieldSetting;
import org.seasar.codegen.exception.InternalGenerateException;

public class MaxDB implements Dbms {

    final String stringTypeName = "string";

    final String numberTypeName = "num";

    final String dateTypeName = "date";

    final String timeTypeName = "time";

    final String timestampTypeName = "timestamp";

    final String booleanTypeName = "bool";

    final String blobTypeName = "blob";

    private LanguageDataTypeSelectHelper langSelectUtil;

    private Map<String, String> typeToType = new HashMap<String, String>();

    public MaxDB() {
        typeToType.put("BIGINT", numberTypeName);
        typeToType.put("INTEGER", numberTypeName);
        typeToType.put("VARCHAR", stringTypeName);
        typeToType.put("LONG", stringTypeName);
        typeToType.put("DATE", dateTypeName);
        typeToType.put("NUMBER", numberTypeName);
        typeToType.put("SMALLINT", numberTypeName);
    }

    public String convDBTypeToDataType(String typeName) {
        return (String) typeToType.get(typeName);
    }

    public DataType selectBestDataType(FieldSetting fieldSetting)
            throws InternalGenerateException {
        String lowTypeName = fieldSetting.getTypeName().toLowerCase();
        DataType dataType = null;
        if (lowTypeName.startsWith(stringTypeName)) {
            dataType = getStringDataType(fieldSetting);
        } else if (lowTypeName.startsWith(numberTypeName)) {
            dataType = getNumberTypeName(fieldSetting);
        } else if (lowTypeName.startsWith(dateTypeName)) {
            dataType = getTimestampTypeName(fieldSetting);
        } else if (lowTypeName.startsWith(timeTypeName)) {
            dataType = getTimestampTypeName(fieldSetting);
        } else if (lowTypeName.startsWith(timestampTypeName)) {
            dataType = getTimestampTypeName(fieldSetting);
        } else if (lowTypeName.startsWith(booleanTypeName)) {
            dataType = getBooleanTypeName(fieldSetting);
            // } else if(lowTypeName.startsWith(blobTypeName)) {
            // dataType = getStringDataType(fieldSetting);
        } else {
            throw new InternalGenerateException(fieldSetting.getTypeName());
        }

        return dataType;
    }

    protected DataType getStringDataType(FieldSetting fieldSetting) {
        fieldSetting.setRdbType("VARCHAR");
        DataType dataType = new DataType(fieldSetting);
        dataType.setLangDataType(langSelectUtil.getDataType("String",
                fieldSetting.isNotNull(), fieldSetting.getFieldDefault()));
        return dataType;
    }

    protected DataType getNumberTypeName(FieldSetting fieldSetting) {
        if (fieldSetting.getPointNumber() > 0) {
            return getDoubleTypeName(fieldSetting);
        }
        if (fieldSetting.getColmnSize() == 2) {
            return getBooleanTypeName(fieldSetting);
        }
        return getLongTypeName(fieldSetting);

    }

    protected DataType getDoubleTypeName(FieldSetting fieldSetting) {
        fieldSetting.setRdbType("NUMBER");
        if (fieldSetting.isNotNull()) {
            fieldSetting.setRdbType("NUMBER");
        }
        DataType dataType = new DataType(fieldSetting);
        dataType.setLangDataType(langSelectUtil.getDataType("Double",
                fieldSetting.isNotNull(), fieldSetting.getFieldDefault()));
        return dataType;
    }

    protected DataType getLongTypeName(FieldSetting fieldSetting) {
        fieldSetting.setRdbType("INTEGER");
        if (fieldSetting.isNotNull()) {
            fieldSetting.setRdbType("INTEGER");
        }
        DataType dataType = new DataType(fieldSetting);
        dataType.setLangDataType(langSelectUtil.getDataType("Long",
                fieldSetting.isNotNull(), fieldSetting.getFieldDefault()));
        return dataType;
    }

    protected DataType getTimestampTypeName(FieldSetting fieldSetting) {
        fieldSetting.setRdbType("DATE");
        DataType dataType = new DataType(fieldSetting);
        dataType.setLangDataType(langSelectUtil.getDataType("Date",
                fieldSetting.isNotNull(), fieldSetting.getFieldDefault()));
        return dataType;
    }

    protected DataType getBooleanTypeName(FieldSetting fieldSetting) {
        fieldSetting.setRdbType("NUMBER");
        fieldSetting.setColmnSize(2);
        if (fieldSetting.isNotNull()) {
            fieldSetting.setRdbType("NUMBER");
        }
        DataType dataType = new DataType(fieldSetting);
        dataType.setLangDataType(langSelectUtil.getDataType("Boolean",
                fieldSetting.isNotNull(), fieldSetting.getFieldDefault()));
        return dataType;
    }

    public LanguageDataTypeSelectHelper getLangSelectUtil() {
        return langSelectUtil;
    }

    public void setLangSelectUtil(LanguageDataTypeSelectHelper langSelectUtil) {
        this.langSelectUtil = langSelectUtil;
    }

    public String getSuffix() {
        return "maxdb";
    }

}
