/*
 * Copyright 2006-2011 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.codegen.lib.impl;

import java.util.ArrayList;
import java.util.List;

import junit.framework.TestCase;

import org.seasar.codegen.lib.BindValue;
import org.seasar.codegen.lib.Condition;
import org.seasar.codegen.lib.ConditionResult;
import org.seasar.codegen.lib.Operator;
import org.seasar.codegen.lib.Order;
import org.seasar.codegen.lib.exception.NullConditionValueException;
import org.seasar.dao.pager.DefaultPagerCondition;

/**
 * @author azusa
 * 
 */
public class ConditionElementImplTest extends TestCase {

    /**
     * 
     */
    private static final String COLUMN = "bbb";

    /**
     * 
     */
    private static final String ALIAS = "aaa";

    // private EmpTableCondition empCondition = new EmpTableCondition();

    private ConditionResult conditionResult = new ConditionResultImpl();

    private ConditionElementImpl<Integer> conditionElement = new ConditionElementImpl<Integer>(conditionResult,
            new ColumnImpl(ALIAS, COLUMN, NullCondition.class));

    protected void setUp() throws Exception {
        super.setUp();
    }

    public void test昇順() {
        conditionElement.addOrderAsc();
        List<Order> orderList = conditionResult.getOrderList();
        assertEquals(1, orderList.size());
        assertEquals(ALIAS, orderList.get(0).getColumn().getAlias());
        assertTrue(orderList.get(0).isAsc());
    }

    public void test降順() {
        conditionElement.addOrderDesc();
        List<Order> orderList = conditionResult.getOrderList();
        assertEquals(1, orderList.size());
        assertEquals(ALIAS, orderList.get(0).getColumn().getAlias());
        assertFalse(orderList.get(0).isAsc());

    }

    public void testEqual() {
        conditionElement.equal(1);
        List<BindValue> bindValues = conditionResult.getBindValues();
        BindValue bv = bindValues.get(0);
        assertEquals(1, bv.getArgs()[0]);
        assertEquals(COLUMN, bv.getColumn().getName());
        assertEquals(Operator.EQUAL, bv.getOperator());
    }

    /**
     * {@link org.seasar.codegen.lib.impl.ConditionElementImpl#in(java.lang.Object[])}
     * のためのテスト・メソッド。
     */
    public void testInObjectArray() {
        conditionElement.in(1, 2);
        List<BindValue> bindValues = conditionResult.getBindValues();
        BindValue bv = bindValues.get(0);
        assertEquals(1, bv.getArgs()[0]);
        assertEquals(2, bv.getArgs()[1]);
        assertEquals(COLUMN, bv.getColumn().getName());
        assertEquals(Operator.IN, bv.getOperator());

    }

    /**
     * {@link org.seasar.codegen.lib.impl.ConditionElementImpl#in(java.util.List)}
     * のためのテスト・メソッド。
     */
    public void testInList() {
        List<Integer> list = new ArrayList<Integer>();
        list.add(1);
        list.add(2);
        conditionElement.in(list);
        List<BindValue> bindValues = conditionResult.getBindValues();
        BindValue bv = bindValues.get(0);
        assertEquals(1, bv.getArgs()[0]);
        assertEquals(2, bv.getArgs()[1]);
        assertEquals(COLUMN, bv.getColumn().getName());
        assertEquals(Operator.IN, bv.getOperator());
    }

    /**
     * {@link org.seasar.codegen.lib.impl.ConditionElementImpl#isNotNull()}
     * のためのテスト・メソッド。
     */
    public void testIsNotNull() {
        conditionElement.isNotNull();
        List<BindValue> bindValues = conditionResult.getBindValues();
        BindValue bv = bindValues.get(0);
        assertEquals(COLUMN, bv.getColumn().getName());
        assertEquals(Operator.NOT_NULL, bv.getOperator());
        assertEquals(0, bv.getArgs().length);
    }

    /**
     * {@link org.seasar.codegen.lib.impl.ConditionElementImpl#isNull()}
     * のためのテスト・メソッド。
     */
    public void testIsNull() {
        conditionElement.isNull();
        List<BindValue> bindValues = conditionResult.getBindValues();
        BindValue bv = bindValues.get(0);
        assertEquals(COLUMN, bv.getColumn().getName());
        assertEquals(Operator.NULL, bv.getOperator());
        assertEquals(0, bv.getArgs().length);

    }

    /**
     * {@link org.seasar.codegen.lib.impl.ConditionElementImpl#less(java.lang.Object)}
     * のためのテスト・メソッド。
     */
    public void testLess() {
        conditionElement.less(1);
        List<BindValue> bindValues = conditionResult.getBindValues();
        BindValue bv = bindValues.get(0);
        assertEquals(COLUMN, bv.getColumn().getName());
        assertEquals(Operator.LESS, bv.getOperator());
        assertEquals(1, bv.getArgs()[0]);
    }

    /**
     * {@link org.seasar.codegen.lib.impl.ConditionElementImpl#lessThan(java.lang.Object)}
     * のためのテスト・メソッド。
     */
    public void testLessThan() {
        conditionElement.lessThan(1);
        List<BindValue> bindValues = conditionResult.getBindValues();
        BindValue bv = bindValues.get(0);
        assertEquals(COLUMN, bv.getColumn().getName());
        assertEquals(Operator.LESSTHAN, bv.getOperator());
        assertEquals(1, bv.getArgs()[0]);
    }

    /**
     * {@link org.seasar.codegen.lib.impl.ConditionElementImpl#more(java.lang.Object)}
     * のためのテスト・メソッド。
     */
    public void testMore() {
        conditionElement.more(1);
        List<BindValue> bindValues = conditionResult.getBindValues();
        BindValue bv = bindValues.get(0);
        assertEquals(COLUMN, bv.getColumn().getName());
        assertEquals(Operator.MORE, bv.getOperator());
        assertEquals(1, bv.getArgs()[0]);
    }

    /**
     * {@link org.seasar.codegen.lib.impl.ConditionElementImpl#moreThan(java.lang.Object)}
     * のためのテスト・メソッド。
     */
    public void testMoreThan() {
        conditionElement.moreThan(1);
        List<BindValue> bindValues = conditionResult.getBindValues();
        BindValue bv = bindValues.get(0);
        assertEquals(COLUMN, bv.getColumn().getName());
        assertEquals(Operator.MORETHAN, bv.getOperator());
        assertEquals(1, bv.getArgs()[0]);
    }

    /**
     * {@link org.seasar.codegen.lib.impl.ConditionElementImpl#not(java.lang.Object)}
     * のためのテスト・メソッド。
     */
    public void testNot() {
        conditionElement.not(1);
        List<BindValue> bindValues = conditionResult.getBindValues();
        BindValue bv = bindValues.get(0);
        assertEquals(COLUMN, bv.getColumn().getName());
        assertEquals(Operator.NOT, bv.getOperator());
        assertEquals(1, bv.getArgs()[0]);
    }

    /**
     * {@link org.seasar.codegen.lib.impl.ConditionElementImpl#notIn(java.lang.Object[])}
     * のためのテスト・メソッド。
     */
    public void testNotInObjectArray() {
        conditionElement.notIn(1, 2);
        List<BindValue> bindValues = conditionResult.getBindValues();
        BindValue bv = bindValues.get(0);
        assertEquals(COLUMN, bv.getColumn().getName());
        assertEquals(Operator.NOTIN, bv.getOperator());
        assertEquals(1, bv.getArgs()[0]);
        assertEquals(2, bv.getArgs()[1]);

    }

    /**
     * {@link org.seasar.codegen.lib.impl.ConditionElementImpl#notIn(java.util.List)}
     * のためのテスト・メソッド。
     */
    public void testNotInList() {
        List<Integer> list = new ArrayList<Integer>();
        list.add(1);
        list.add(2);
        conditionElement.notIn(list);
        List<BindValue> bindValues = conditionResult.getBindValues();
        BindValue bv = bindValues.get(0);
        assertEquals(COLUMN, bv.getColumn().getName());
        assertEquals(Operator.NOTIN, bv.getOperator());
        assertEquals(1, bv.getArgs()[0]);
        assertEquals(2, bv.getArgs()[1]);
    }

    public void testNull() {
        try {
            conditionElement.equal(null);
            fail();
        } catch (NullConditionValueException e) {
            System.out.println(e.getMessage());
            assertTrue(true);
        }

    }

    private static class NullCondition extends DefaultPagerCondition implements Condition {

        /*
         * (非 Javadoc)
         * 
         * @see org.seasar.codegen.lib.Condition#addQuery(java.lang.String,
         * java.lang.Object[])
         */
        public void addQuery(String query, Object... args) {
        }

        /*
         * (非 Javadoc)
         * 
         * @see org.seasar.codegen.lib.Condition#getConditionResult()
         */
        public ConditionResult getConditionResult() {
            return null;
        }

    }
}
