/*
 * Copyright 2004-2006 H2 Group. Licensed under the H2 License, Version 1.0 (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package org.h2.store;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.sql.SQLException;

import org.h2.engine.Constants;
import org.h2.message.Message;
import org.h2.security.SecureFileStore;
import org.h2.util.ByteUtils;
import org.h2.util.FileUtils;

public class FileStore {

    public static final int HEADER_LENGTH = 3 * Constants.FILE_BLOCK_SIZE;

    protected String name;
    protected DataHandler database;
    private byte[] magic;

    private RandomAccessFile file;
    private long filePos;

    public static FileStore open(DataHandler database, String name, byte[] magic, String cipher, byte[] key) throws SQLException {
        FileStore store;
        if(FileUtils.isInMemory(name)) {
            store = new MemoryFileStore(database, name, magic);
        } else if(cipher == null) {
            store = new FileStore(database, name, magic);
        } else {
            store = new SecureFileStore(database, name, magic, cipher, key);
        }
        return store;
    }

    protected FileStore(DataHandler database, String name, byte[] magic) throws SQLException {
        this.database = database;
        this.name = name;
        this.magic = magic;
        try {
            FileUtils.createDirs(name);
            File f = new File(name);
            if(f.exists() && !f.canWrite()) {
                file = FileUtils.openRandomAccessFile(name, "r");
            } else {
                // file = new RandomAccessFile(name, "rws");
                file = FileUtils.openRandomAccessFile(name, "rw");
            }
        } catch(IOException e) {
            throw Message.convert(e);
        }
    }

    protected FileStore(DataHandler database, byte[] magic) {
        this.database = database;
        this.magic = magic;
    }

    protected byte[] generateSalt() {
        return magic;
    }

    protected void initKey(byte[] salt) {
        // do nothing
    }

    protected void checkWritingAllowed() throws SQLException {
        if(database != null) {
            database.checkWritingAllowed();
        }
    }

    protected void checkPowerOff() throws SQLException {
        if(database != null) {
            database.checkPowerOff();
        }
    }

    public void init() throws SQLException {
        int len = Constants.FILE_BLOCK_SIZE;
        byte[] salt;
        if(length() < HEADER_LENGTH) {
            // write unencrypted
            writeDirect(magic, 0, len);
            salt = generateSalt();
            writeDirect(salt, 0, len);
            initKey(salt);
            // write (maybe) encrypted
            write(magic, 0, len);
        } else {
            // write unencrypted
            seek(0);
            byte[] buff = new byte[len];
            readFullyDirect(buff, 0, len);
            if(ByteUtils.compareNotNull(buff, magic) != 0) {
                throw Message.getSQLException(Message.FILE_VERSION_ERROR_1, name);
            }
            salt = new byte[len];
            readFullyDirect(salt, 0, len);
            initKey(salt);
            // read (maybe) encrypted
            readFully(buff, 0, Constants.FILE_BLOCK_SIZE);
            if(ByteUtils.compareNotNull(buff, magic) != 0) {
                throw Message.getSQLException(Message.FILE_ENCRYPTION_ERROR);
            }
        }
    }

    public void close() throws IOException {
        if(file != null) {
            try {
                file.close();
            } finally {
                file = null;
                name = null;
            }
        }
    }

    public void closeAndDeleteSilently() {
        if(file != null) {
            try {
                file.close();
            } catch(IOException e) {
                file = null;
            }
            try {
                FileUtils.delete(name);
            } catch (SQLException e) {
                // TODO log such errors?
            }
            name = null;
        }
    }

    protected void readFullyDirect(byte[] b, int off, int len) throws SQLException {
        readFully(b, off, len);
    }

    public void readFully(byte[] b, int off, int len) throws SQLException {
        if(Constants.CHECK && len < 0) {
            throw Message.internal("read len "+len);
        }
        if(Constants.CHECK && len % Constants.FILE_BLOCK_SIZE != 0) {
            throw Message.internal("unaligned read "+name+" len "+len);
        }
        checkPowerOff();
        try {
            file.readFully(b, off, len);
        } catch (IOException e) {
            throw Message.convert(e);
        }
        filePos += len;
    }

    public void seek(long pos) throws SQLException {
        if(Constants.CHECK && pos % Constants.FILE_BLOCK_SIZE != 0) {
            throw Message.internal("unaligned seek "+name+" pos "+pos);
        }
        try {
            if(pos != filePos) {
                file.seek(pos);
                filePos = pos;
            }
        } catch (IOException e) {
            throw Message.convert(e);
        }
    }

    protected void writeDirect(byte[] b, int off, int len) throws SQLException {
        write(b, off, len);
    }

    public void write(byte[] b, int off, int len) throws SQLException {
        if(Constants.CHECK && len < 0) {
            throw Message.internal("read len "+len);
        }
        if(Constants.CHECK && len % Constants.FILE_BLOCK_SIZE != 0) {
            throw Message.internal("unaligned write "+name+" len "+len);
        }
        checkWritingAllowed();
        checkPowerOff();
        try {
            file.write(b, off, len);
        } catch (IOException e) {
            if(freeUpDiskSpace()) {
                try {
                    file.write(b, off, len);
                } catch (IOException e2) {
                    throw Message.convert(e2);
                }
            } else {
                throw Message.convert(e);
            }
        }
        filePos += len;
    }

    private boolean freeUpDiskSpace() throws SQLException {
        if(database == null) {
            return false;
        }
        database.freeUpDiskSpace();
        return true;
    }

    public void setLength(long newLength) throws SQLException {
        if(Constants.CHECK && newLength % Constants.FILE_BLOCK_SIZE != 0) {
            throw Message.internal("unaligned setLength "+name+" pos "+newLength);
        }
        checkPowerOff();
        checkWritingAllowed();
        try {
            FileUtils.setLength(file, newLength);
        } catch (IOException e) {
            if(freeUpDiskSpace()) {
                try {
                    FileUtils.setLength(file, newLength);
                } catch (IOException e2) {
                    throw Message.convert(e2);
                }
            } else {
                throw Message.convert(e);
            }
        }
    }

    public long length() throws SQLException {
        try {
            if(Constants.CHECK && file.length() % Constants.FILE_BLOCK_SIZE != 0) {
                long newLength = file.length() + Constants.FILE_BLOCK_SIZE - (file.length() % Constants.FILE_BLOCK_SIZE);
                FileUtils.setLength(file, newLength);
                throw Message.internal("unaligned file length "+name+" len "+file.length());
            }
            return file.length();
        } catch (IOException e) {
            throw Message.convert(e);
        }
    }

    public long getFilePointer() throws SQLException {
        if(Constants.CHECK2) {
            try {
                if(file.getFilePointer() != filePos) {
                    throw Message.internal();
                }
            } catch (IOException e) {
                throw Message.convert(e);
            }
        }
        return filePos;
    }

    public void sync() {
        try {
            file.getFD().sync();
        } catch(IOException e) {
            // TODO log exception
        }
    }

}
