/*
 * Copyright 2004-2006 H2 Group. Licensed under the H2 License, Version 1.0 (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package org.h2.security;

import java.sql.SQLException;

import org.h2.engine.Constants;
import org.h2.message.Message;
import org.h2.store.DataHandler;
import org.h2.store.FileStore;
import org.h2.util.RandomUtils;

public class SecureFileStore extends FileStore {

    // to slow down dictionary attacks
    private static final int SHA_ITERATIONS = 1024;
    
    private byte[] key;
    private BlockCipher cipher;
    private BlockCipher cipherForInitVector;
    private byte[] buffer = new byte[4];
    private long pos;
    private byte[] bufferForInitVector;

    public SecureFileStore(DataHandler database, String name, byte[] magic, String cipher, byte[] key) throws SQLException {
        super(database, name, magic);
        this.key = key;
        if ("XTEA".equalsIgnoreCase(cipher)) {
            this.cipher = new XTEA();
            this.cipherForInitVector = new XTEA();
        } else if ("AES".equalsIgnoreCase(cipher)) {
            this.cipher = new AES();
            this.cipherForInitVector = new AES();
        } else {
            throw Message.getSQLException(Message.UNSUPPORTED_CIPHER, cipher);
        }
        bufferForInitVector = new byte[Constants.FILE_BLOCK_SIZE];
    }

    protected byte[] generateSalt() {
        return RandomUtils.getSecureBytes(Constants.FILE_BLOCK_SIZE);
    }

    protected void initKey(byte[] salt) {
        SHA256 sha = new SHA256();
        key = sha.getHashWithSalt(key, salt);
        for (int i = 0; i < SHA_ITERATIONS; i++) {
            key = sha.getHash(key);
        }
        cipher.setKey(key);
        key = sha.getHash(key);
        cipherForInitVector.setKey(key);
    }

    protected void writeDirect(byte[] b, int off, int len) throws SQLException {
        super.write(b, off, len);
        pos += len;
    }

    public void write(byte[] b, int off, int len) throws SQLException {
        if (buffer.length < b.length) {
            buffer = new byte[len];
        }
        System.arraycopy(b, off, buffer, 0, len);
        xorInitVector(buffer, 0, len, pos);
        cipher.encrypt(buffer, 0, len);
        super.write(buffer, 0, len);
        pos += len;
    }

    protected void readFullyDirect(byte[] b, int off, int len) throws SQLException {
        super.readFully(b, off, len);
        pos += len;
    }

    public void readFully(byte[] b, int off, int len) throws SQLException {
        super.readFully(b, off, len);
        cipher.decrypt(b, off, len);
        xorInitVector(b, off, len, pos);
        pos += len;
    }

    public void seek(long x) throws SQLException {
        this.pos = x;
        super.seek(x);
    }
    
    public void setLength(long newLength) throws SQLException {
        long oldPos = pos;
        byte[] buff = new byte[Constants.FILE_BLOCK_SIZE];
        long length = length();
        if(newLength > length) {
            seek(length);
            for(long i = length; i<newLength; i+= Constants.FILE_BLOCK_SIZE) {
                write(buff, 0, Constants.FILE_BLOCK_SIZE);
            }
            seek(oldPos);
        } else {
            super.setLength(newLength);
        }
    }
    
    private void xorInitVector(byte[] b, int off, int len, long pos) {
        byte[] iv = bufferForInitVector;
        int block = (int)(pos >>> 3);
        while(len > 0) {
            iv[0] = (byte) ((block >> 24) & 0xff);
            iv[1] = (byte) ((block >> 16) & 0xff);
            iv[2] = (byte) ((block >> 8) & 0xff);
            iv[3] = (byte) (block & 0xff);
            for(int i=0; i<Constants.FILE_BLOCK_SIZE; i++) {
                iv[i] = 0;
            }
            cipherForInitVector.encrypt(iv, 0, Constants.FILE_BLOCK_SIZE);
            for(int i=0; i<Constants.FILE_BLOCK_SIZE; i++) {
                b[off + i] ^= iv[i];
            }
            off += Constants.FILE_BLOCK_SIZE;
            len -= Constants.FILE_BLOCK_SIZE;
            block++;
        }
    }
    
}
