/*
 * Copyright 2006-2011 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.codegen.impl;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.InputStream;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.seasar.codegen.convert.FKNameConverter;
import org.seasar.codegen.element.Field;
import org.seasar.codegen.element.LinkTable;
import org.seasar.codegen.element.PrimaryKey;
import org.seasar.codegen.element.Table;
import org.seasar.codegen.util.ExcelUtil;
import org.seasar.framework.util.FileInputStreamUtil;
import org.seasar.framework.util.InputStreamUtil;
import org.seasar.framework.util.StringUtil;

/**
 * 1シート複数テーブルの Excel ファイルを読み込みます。
 * 
 * @author glad
 */
public class ExcelImportCodeDataSingleSheet extends ExcelImportCodeDataSupport {

    protected int entityNameCol;

    protected int tableNameCol;

    public ExcelImportCodeDataSingleSheet() {
        entityNameCol = 0;
        tableNameCol = 1;

        attributeNameCol = 2;
        columnNameCol = 3;
        dataTypeCol = 4;
        notNullCol = 5;
        primaryKeyCol = 6;
        foreignKeyCol = 7;
        defaultValueCol = 8;

        relationTypeCol = 9;
        parentTableCol = 10;
        parentColumnCol = 11;
        cardinalityCol = 12;
        parentPropertyCol = 13;
        childPropertyCol = 14;
    }

    public Map<String, Table> readCodeData(File srcFile) {
        InputStream in = FileInputStreamUtil.create(srcFile);
        try {
            in = new BufferedInputStream(in);
            HSSFWorkbook book = ExcelUtil.createWorkbook(in);
            HSSFSheet sheet = book.getSheetAt(0);
            Map<String, Table> tableMap = readCodeData(sheet);
            return tableMap;
        } finally {
            InputStreamUtil.close(in);
        }
    }

    protected Map<String, Table> readCodeData(HSSFSheet sheet) {
        Map<String, Table> tableMap = new LinkedHashMap<String, Table>();
        Table table = null;
        for (int i = 1; i <= sheet.getLastRowNum(); ++i) {
            HSSFRow row = sheet.getRow(i);
            table = processRow(row, table, tableMap);
        }
        setupChildLinks(tableMap);
        return tableMap;
    }

    protected Table processRow(HSSFRow row, Table table,
            Map<String, Table> tableMap) {
        String tableName = getString(row, tableNameCol);
        if (!StringUtil.isEmpty(tableName)) {
            table = new Table();
            table.setTableName(tableName);
            tableMap.put(table.getTableName(), table);
        }
        Field field = getField(row, table);
        if (field != null) {
            table.addTableField(field);

            PrimaryKey primaryKey = getPrimaryKey(row, field);
            if (primaryKey != null) {
                table.addPrimaryKey(primaryKey);
            }
            LinkTable parentLink = getParentLink(row, field);
            if (parentLink != null) {
                String relName = converter.convertParent(table, parentLink);
                table.addLinkTable(relName, parentLink);
            }
        }
        return table;
    }

    public void setConverter(FKNameConverter converter) {
        this.converter = converter;
    }

}
