/*
 * Copyright 2004-2006 H2 Group. Licensed under the H2 License, Version 1.0 (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package org.h2.tools;

import java.io.*;
import java.sql.*;
import java.util.ArrayList;

import org.h2.util.IOUtils;
import org.h2.util.StringUtils;

/**
 * A facility to read from and write to CSV (comma separated values) files.
 *
 */
public class Csv implements SimpleRowSource {
    
    private String charset = StringUtils.getDefaultCharset();
    private int bufferSize = 8 * 1024;
    private String[] columnNames;
    private char fieldSeparatorRead = ',';
    private char commentLineStart = '#';
    private String fieldSeparatorWrite = ",";
    private String rowSeparatorWrite = null;
    private char fieldDelimiter = '\"';
    private char escapeCharacter = '\"';
    private String fileName;
    private InputStream in;
    private Reader reader;
    private FileOutputStream out;
    private PrintWriter writer;
    private int back;
    private boolean endOfLine, endOfFile;
    
    /**
     * Writes the result set to a file in the CSV format
     * @param fileName
     * @param rs the result set
     * @param charset the charset or null to use UTF-8
     * @throws SQLException
     */
    public static void write(String fileName, ResultSet rs, String charset) throws SQLException {
        ResultSetMetaData meta = rs.getMetaData();
        Csv csv = new Csv(fileName, charset);
        try {
            csv.initWrite();
            int columnCount = meta.getColumnCount();
            String[] row = new String[columnCount];
            for(int i=0; i<columnCount; i++) {
                row[i] = meta.getColumnLabel(i + 1);
            }
            csv.writeRow(row);
            while(rs.next()) {
                for(int i=0; i<columnCount; i++) {
                    row[i] = rs.getString(i + 1);
                }
                csv.writeRow(row);
            }
            csv.close();
        } catch(IOException e) {
            throw csv.convertException("IOException writing file " + fileName, e);
        }
        rs.close();
    }
    
    /**
     * Writes the result set of a query to a file in the CSV format.
     * 
     * @param conn the connection
     * @param fileName the file name
     * @param sql the query
     * @param charset the charset or null to use UTF-8
     * @throws SQLException
     */
    public static void write(Connection conn, String fileName, String sql, String charset) throws SQLException {
        Statement stat = conn.createStatement();
        ResultSet rs = stat.executeQuery(sql);
        write(fileName, rs, charset);
        stat.close();
    }

    /**
     * Reads from the CSV file and returns a result set.
     * The rows in the result set are created on demand, 
     * that means the file is kept open until all rows are read
     * or the result set is closed.
     * 
     * @param fileName the file name
     * @param columnNames or null if the column names should be read from the CSV file
     * @param charset the charset or null to use UTF-8
     * @return the result set
     * @throws SQLException
     */
    public static ResultSet read(String fileName, String[] columnNames, String charset) throws SQLException {
        Csv csv = new Csv(fileName, charset);
        try {
            csv.columnNames = columnNames;
            csv.initRead();
            SimpleResultSet result = new SimpleResultSet(csv);
            columnNames = csv.columnNames;
            columnNames = normalizeColumnNames(columnNames);
            for(int i=0; i<columnNames.length; i++) {
                result.addColumn(columnNames[i], Types.VARCHAR, 255, 0);
            }
            return result;
        } catch(IOException e) {
            throw csv.convertException("IOException reading file " + fileName, e);
        }
    }
    
    /**
     * Reads CSV data from a reader and returns a result set.
     * The rows in the result set are created on demand, 
     * that means the reader is kept open until all rows are read
     * or the result set is closed.
     * 
     * @param reader the reader
     * @param columnNames or null if the column names should be read from the CSV file
     * @return the result set
     * @throws SQLException
     */
    public static ResultSet read(Reader reader, String[] columnNames) throws SQLException {
        Csv csv = new Csv(null, null);
        try {
            csv.columnNames = columnNames;
            csv.reader = reader;
            csv.initRead();
            SimpleResultSet result = new SimpleResultSet(csv);
            columnNames = csv.columnNames;
            columnNames = normalizeColumnNames(columnNames);
            for(int i=0; i<columnNames.length; i++) {
                result.addColumn(columnNames[i], Types.VARCHAR, 255, 0);
            }
            return result;
        } catch(IOException e) {
            throw csv.convertException("IOException", e);
        }
    }
    
    private static String[] normalizeColumnNames(String[] columnNames) {
        for(int i=0; i<columnNames.length; i++) {
            String x = columnNames[i];
            if(x == null || x.length()==0) {
                x = "C" + (i+1);
            }
            for(int j=0; j<i; j++) {
                String y = columnNames[j];
                if(x.equals(y)) {
                    x = x + "1";
                    j = -1;
                }
            }
            columnNames[i] = x;
        }
        return columnNames;
    }

    private Csv(String fileName, String charset) {
        this.fileName = fileName;
        if(charset != null) {
            this.charset = charset;
        }
    }
    
    private void initWrite() throws IOException {
        if(writer == null) {
            try {
                out = new FileOutputStream(fileName);
                BufferedOutputStream o = new BufferedOutputStream(out, bufferSize);
                writer = new PrintWriter(new OutputStreamWriter(o, charset));
                // TODO performance: what is faster? one, two, or both?
                // writer = new PrintWriter(new BufferedWriter(new OutputStreamWriter(out, encoding), bufferSize));
            } catch(IOException e) {
                close();
                throw e;
            }
        }
        if(columnNames != null) {
            writeHeader();            
        }
    }
    
    private void writeHeader() {
        for(int i=0; i<columnNames.length; i++) {
            if(i>0) {
                writer.print(fieldSeparatorRead);
            }
            writer.print(columnNames[i]);
        }
        writer.println();
    }
    
    private void writeRow(String[] values) {
        for(int i=0; i<values.length; i++) {
            if(i>0) {
                if(fieldSeparatorWrite != null) {
                    writer.print(fieldSeparatorWrite);
                }
            }
            String s = values[i];
            if(s != null) {
                if(escapeCharacter != 0) {
                    if(fieldDelimiter != 0) {
                        writer.print(fieldDelimiter);
                    }
                    writer.print(escape(s));
                    if(fieldDelimiter != 0) {
                        writer.print(fieldDelimiter);
                    }
                } else {
                    writer.print(s);
                }
            }
        }
        if(rowSeparatorWrite != null) {
            writer.print(rowSeparatorWrite);
        }
        writer.println();
    }

    private String escape(String data) {
        if(data.indexOf(fieldDelimiter) < 0) {
            if(escapeCharacter == fieldDelimiter || data.indexOf(escapeCharacter) < 0) {
                return data;
            }
        }
        StringBuffer buff = new StringBuffer();
        for(int i=0; i<data.length(); i++) {
            char ch = data.charAt(i);
            if(ch == fieldDelimiter || ch == escapeCharacter) {
                buff.append(escapeCharacter);
            }
            buff.append(ch);
        }
        return buff.toString();
    }
    
    private void initRead() throws IOException {
        if(reader == null) {
            try {
                in = new FileInputStream(fileName);
                BufferedInputStream i = new BufferedInputStream(in, bufferSize);
                reader = new InputStreamReader(i, charset);
                // TODO what is faster, 1, 2, 1+2
                //reader = new BufferedReader(new InputStreamReader(in, encoding), bufferSize);
            } catch(IOException e) {
                close();
                throw e;
            }
        }
        if(columnNames == null) {
            readHeader();
        }
    }
    
    private void readHeader() throws IOException {
        ArrayList list = new ArrayList();
        while(true) {
            String v = readValue();
            if(v==null) {
                if(endOfLine) {
                    if(endOfFile || list.size()>0) {
                        break;
                    }
                } else {
                    list.add("COLUMN" + list.size());
                }
            } else {
                list.add(v);
            }
        }
        columnNames = new String[list.size()];
        list.toArray(columnNames);
    }
    
    private void pushBack(int ch) {
        back = ch;
    }
    
    private int readChar() throws IOException {
        int ch = back;
        if(ch != -1) {
            back = -1;
            return ch;
        } else if(endOfFile) {
            return -1;
        }
        ch = reader.read();
        if(ch < 0) {
            endOfFile = true;
            close();
        }
        return ch;
    }
    
    private String readValue() throws IOException {
        endOfLine = false;
        String value = null;
        while(true) {
            int ch = readChar();
            if(ch < 0 || ch == '\r' || ch == '\n') {
                endOfLine = true;
                break;
            } else if(ch <= ' ') {
                // ignore spaces
                continue;
            } else if(ch == fieldSeparatorRead) {
                break;
            } else if(ch == commentLineStart) {
                while(true) {
                    ch = readChar();
                    if(ch < 0 || ch == '\r' || ch == '\n') {
                        break;
                    }
                }
                endOfLine = true;
                break;
            } else if(ch == fieldDelimiter) {
                StringBuffer buff = new StringBuffer();
                boolean containsEscape = false;
                while(true) {
                    ch = readChar();
                    if(ch < 0) {
                        return buff.toString();
                    } else if(ch == fieldDelimiter) {
                        ch = readChar();
                        if(ch == fieldDelimiter) {
                            buff.append((char)ch);
                        } else {
                            pushBack(ch);
                            break;
                        }
                    } else if(ch == escapeCharacter) {
                        buff.append((char)ch);
                        ch = readChar();
                        if(ch < 0) {
                            break;
                        }
                        containsEscape = true;
                        buff.append((char)ch);
                    } else {
                        buff.append((char)ch);
                    }
                }
                value = buff.toString();
                if(containsEscape) {
                    value = unEscape(value);
                }
                while(true) {
                    ch = readChar();
                    if(ch < 0) {
                        break;
                    } else if(ch == ' ' || ch == '\t') {
                        // ignore
                    } else if(ch == fieldSeparatorRead) {
                        break;
                    } else if(ch == '\r' || ch == '\n') {
                        pushBack(ch);
                        endOfLine = true;
                        break;
                    } else {
                        pushBack(ch);
                        break;
                    }
                }
                break;
            } else {
                StringBuffer buff = new StringBuffer();
                buff.append((char)ch);
                while(true) {
                    ch = readChar();
                    if(ch == fieldSeparatorRead) {
                        break;
                    } else if(ch == '\r' || ch == '\n') {
                        pushBack(ch);
                        endOfLine = true;
                        break;
                    } else if(ch < 0) {
                        break;
                    }
                    buff.append((char)ch);
                }
                value = buff.toString().trim();
                break;
            }
        }
        return value;
    }

    private String unEscape(String s) {
        StringBuffer buff = new StringBuffer();
        int start = 0;
        while(true) {
            int idx = s.indexOf(escapeCharacter, start);
            if(idx < 0) {
                break;
            }
            buff.append(s.toCharArray(), start, idx);
            start = idx + 1;
        }
        buff.append(s.substring(start));
        return buff.toString();
    }

    /**
     * INTERNAL
     */
    public Object[] readRow() throws SQLException {
        if(reader == null) {
            return null;
        }
        String[] row = new String[columnNames.length];
        try {
            for(int i=0;; i++) {
                String v = readValue();
                if(v==null) {
                    if(endOfFile && i==0) {
                        return null;
                    }
                    if(endOfLine) {
                        if(i==0) {
                            // empty line
                            i--;
                            continue;
                        }
                        break;
                    }
                }
                if(i < row.length) {
                    row[i] = v;
                }
            }
        } catch(IOException e) {
            throw convertException("IOException reading from " + fileName, e);
        }
        return row;
    }
    
    private SQLException convertException(String message, Exception e) {
        SQLException s = new SQLException(message, "CSV");
        s.initCause(e);
        return s;
    }

    /**
     * INTERNAL
     */
    public void close() {
        IOUtils.closeSilently(reader);
        reader = null;
        IOUtils.closeSilently(in);
        in = null;
        IOUtils.closeSilently(writer);
        writer = null;
        IOUtils.closeSilently(out);
        out = null;
    }

    /**
     * Override the field separator for writing. The default is ",".
     * @param fieldSeparatorWrite
     */
    public void setFieldSeparatorWrite(String fieldSeparatorWrite) {
        this.fieldSeparatorWrite = fieldSeparatorWrite;
    }

    /**
     * Override the end-of-row marker for writing. The default is null.
     * @param fieldSeparatorWrite
     */
    public void setRowSeparatorWrite(String rowSeparatorWrite) {
        this.rowSeparatorWrite = rowSeparatorWrite;
    }

}
