/*
 * Copyright 2004-2006 H2 Group. Licensed under the H2 License, Version 1.0 (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package org.h2.web;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;

public class FtpData extends Thread {

    private FtpServer server;
    private InetAddress address;
    private ServerSocket serverSocket;
    private volatile Socket socket;
    
    public FtpData(FtpServer server, InetAddress address, ServerSocket serverSocket) throws IOException {
        this.server = server;
        this.address = address;
        this.serverSocket = serverSocket;
    }
    
    public void run() {
        try {
            synchronized(this) {
                Socket s = serverSocket.accept();
                if(s.getInetAddress().equals(address)) {
                    server.trace("Data connected:" + s.getInetAddress() + " expected:" + address);
                    socket = s;
                    notifyAll();
                } else {
                    server.trace("Data REJECTED:" + s.getInetAddress() + " expected:" + address);
                    close();
                }
            }
        } catch(IOException e) {
            e.printStackTrace();
        }
    }
    
    private void waitUntilConnected() {
        while(serverSocket != null && socket == null) {
            try {
                wait();
            } catch(InterruptedException e) {
                // ignore
            }
        }
    }

    public void close() {
        serverSocket = null;
        socket = null;
    }

    public synchronized void send(InputStream in) throws IOException {
        waitUntilConnected();
        try {
            OutputStream out = socket.getOutputStream();
            byte[] buffer = new byte[4 * 1024];
            while(true) {
                int len = in.read(buffer);
                if(len < 0) {
                    break;
                }
                out.write(buffer, 0, len);
            }
            out.close();
            socket.close();
        } finally {
            socket.close();
        }
    }

    public synchronized void receive(OutputStream out) throws IOException {
        waitUntilConnected();
        InputStream in = socket.getInputStream();
        byte[] buffer = new byte[4 * 1024];
        while(true) {
            int len = in.read(buffer);
            if(len < 0) {
                break;
            }
            out.write(buffer, 0, len);
        }
        out.close();
        socket.close();
    }

}
