/*
 * Copyright 2004-2010 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.doma.internal.jdbc.query;

import static org.seasar.doma.internal.util.AssertionUtil.*;

import java.lang.reflect.Method;
import java.sql.Statement;
import java.util.ArrayList;

import org.seasar.doma.internal.jdbc.entity.AbstractPostInsertContext;
import org.seasar.doma.internal.jdbc.entity.AbstractPreInsertContext;
import org.seasar.doma.internal.jdbc.sql.PreparedSql;
import org.seasar.doma.internal.jdbc.sql.PreparedSqlBuilder;
import org.seasar.doma.jdbc.Config;
import org.seasar.doma.jdbc.JdbcException;
import org.seasar.doma.jdbc.SqlKind;
import org.seasar.doma.jdbc.entity.EntityPropertyType;
import org.seasar.doma.jdbc.entity.EntityType;
import org.seasar.doma.jdbc.entity.GeneratedIdPropertyType;
import org.seasar.doma.jdbc.id.IdGenerationConfig;
import org.seasar.doma.message.Message;

/**
 * @author taedium
 * 
 */
public class AutoBatchInsertQuery<E> extends AutoBatchModifyQuery<E> implements
        BatchInsertQuery {

    protected GeneratedIdPropertyType<? super E, E, ?, ?> generatedIdPropertyType;

    protected IdGenerationConfig idGenerationConfig;

    protected boolean batchSupported = true;

    public AutoBatchInsertQuery(EntityType<E> entityType) {
        super(entityType);
    }

    @Override
    public void prepare() {
        assertNotNull(method, config, callerClassName, callerMethodName,
                entities, sqls);
        int size = entities.size();
        if (size == 0) {
            return;
        }
        executable = true;
        executionSkipCause = null;
        currentEntity = entities.get(0);
        preInsert();
        prepareIdAndVersionPropertyTypes();
        prepareOptions();
        prepareTargetPropertyTypes();
        prepareIdValue();
        prepareVersionValue();
        prepareSql();
        entities.set(0, currentEntity);
        for (int i = 1; i < size; i++) {
            currentEntity = entities.get(i);
            preInsert();
            prepareIdValue();
            prepareVersionValue();
            prepareSql();
            entities.set(i, currentEntity);
        }
        currentEntity = null;
        assertEquals(entities.size(), sqls.size());
    }

    protected void preInsert() {
        AutoBatchPreInsertContext<E> context = new AutoBatchPreInsertContext<E>(
                entityType, method, config);
        entityType.preInsert(currentEntity, context);
        if (context.getNewEntity() != null) {
            currentEntity = context.getNewEntity();
        }
    }

    @Override
    protected void prepareIdAndVersionPropertyTypes() {
        super.prepareIdAndVersionPropertyTypes();
        generatedIdPropertyType = entityType.getGeneratedIdPropertyType();
        if (generatedIdPropertyType != null) {
            if (idGenerationConfig == null) {
                idGenerationConfig = new IdGenerationConfig(config, entityType);
                generatedIdPropertyType
                        .validateGenerationStrategy(idGenerationConfig);
                autoGeneratedKeysSupported = generatedIdPropertyType
                        .isAutoGeneratedKeysSupported(idGenerationConfig);
                batchSupported = generatedIdPropertyType
                        .isBatchSupported(idGenerationConfig);
            }
        }
    }

    protected void prepareTargetPropertyTypes() {
        targetPropertyTypes = new ArrayList<EntityPropertyType<E, ?>>(
                entityType.getEntityPropertyTypes().size());
        for (EntityPropertyType<E, ?> p : entityType.getEntityPropertyTypes()) {
            if (!p.isInsertable()) {
                continue;
            }
            if (p.isId()) {
                if (p != generatedIdPropertyType
                        || generatedIdPropertyType
                                .isIncluded(idGenerationConfig)) {
                    targetPropertyTypes.add(p);
                }
                if (generatedIdPropertyType == null
                        && p.getWrapper(currentEntity).get() == null) {
                    throw new JdbcException(Message.DOMA2020,
                            entityType.getName(), p.getName());
                }
                continue;
            }
            if (!isTargetPropertyName(p.getName())) {
                continue;
            }
            targetPropertyTypes.add(p);
        }
    }

    protected void prepareIdValue() {
        if (generatedIdPropertyType != null && idGenerationConfig != null) {
            if (entityType.isImmutable()) {
                E newEntity = generatedIdPropertyType.preInsertAndNewEntity(
                        currentEntity, idGenerationConfig, entityType);
                if (newEntity != null) {
                    currentEntity = newEntity;
                }
            } else {
                generatedIdPropertyType.preInsert(currentEntity,
                        idGenerationConfig);
            }
        }
    }

    protected void prepareVersionValue() {
        if (versionPropertyType != null) {
            if (entityType.isImmutable()) {
                E newEntity = versionPropertyType
                        .setIfNecessaryAndMakeNewEntity(currentEntity, 1,
                                entityType);
                if (newEntity != null) {
                    currentEntity = newEntity;
                }
            } else {
                versionPropertyType.setIfNecessary(currentEntity, 1);
            }
        }
    }

    protected void prepareSql() {
        PreparedSqlBuilder builder = new PreparedSqlBuilder(config,
                SqlKind.BATCH_INSERT);
        builder.appendSql("insert into ");
        builder.appendSql(entityType.getQualifiedTableName());
        builder.appendSql(" (");
        for (EntityPropertyType<E, ?> p : targetPropertyTypes) {
            builder.appendSql(p.getColumnName());
            builder.appendSql(", ");
        }
        builder.cutBackSql(2);
        builder.appendSql(") values (");
        for (EntityPropertyType<E, ?> p : targetPropertyTypes) {
            builder.appendWrapper(p.getWrapper(currentEntity));
            builder.appendSql(", ");
        }
        builder.cutBackSql(2);
        builder.appendSql(")");
        PreparedSql sql = builder.build();
        sqls.add(sql);
    }

    @Override
    public boolean isBatchSupported() {
        return batchSupported;
    }

    @Override
    public void generateId(Statement statement, int index) {
        if (generatedIdPropertyType != null && idGenerationConfig != null) {
            if (entityType.isImmutable()) {
                E newEntity = generatedIdPropertyType.postInsertAndNewEntity(
                        entities.get(index), idGenerationConfig, statement,
                        entityType);
                if (newEntity != null) {
                    entities.set(index, newEntity);
                }
            } else {
                generatedIdPropertyType.postInsert(entities.get(index),
                        idGenerationConfig, statement);
            }
        }
    }

    @Override
    public void complete() {
        for (int i = 0, len = entities.size(); i < len; i++) {
            currentEntity = entities.get(i);
            postInsert();
            entities.set(i, currentEntity);
        }
    }

    protected void postInsert() {
        AutoBatchPostInsertContext<E> context = new AutoBatchPostInsertContext<E>(
                entityType, method, config);
        entityType.postInsert(currentEntity, context);
        if (context.getNewEntity() != null) {
            currentEntity = context.getNewEntity();
        }
    }

    protected static class AutoBatchPreInsertContext<E> extends
            AbstractPreInsertContext<E> {

        public AutoBatchPreInsertContext(EntityType<E> entityType,
                Method method, Config config) {
            super(entityType, method, config);
        }
    }

    protected static class AutoBatchPostInsertContext<E> extends
            AbstractPostInsertContext<E> {

        public AutoBatchPostInsertContext(EntityType<E> entityType,
                Method method, Config config) {
            super(entityType, method, config);
        }
    }
}
