/*
 * Copyright 2006-2011 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.codegen.sample.dao;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.seasar.codegen.sample.condition.EmpTableCondition;
import org.seasar.codegen.sample.condition.EmpTableRelationCondition;
import org.seasar.codegen.sample.entity.EmpTable;
import org.seasar.codegen.sample.entity.EmpTableRelation;
import org.seasar.codegen.unit.CodeGenTestCase;
import org.seasar.dao.tiger.FetchHandler;
import org.seasar.framework.util.InputStreamReaderUtil;
import org.seasar.framework.util.ReaderUtil;
import org.seasar.framework.util.ResourceUtil;

/**
 * @author azusa
 * 
 */
public class EmpTableDaoTest extends CodeGenTestCase {

    private EmpTableDao dao;

    public EmpTableDaoTest(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        include("org/seasar/codegen/sample/dicon/alldao.dicon");
    }

    public void testGetAllEmpTx() throws Exception {
        readXlsAllReplaceDb("DB.xls");
        assertNotNull("1", dao.getAllEmpTable());
    }

    public void testGetEmpTx() throws Exception {
        readXlsAllReplaceDb("DB.xls");
        assertNotNull("1", dao.getEmpTable(7521));
    }

    public void testGetEmpByIds() throws Exception {
        readXlsAllReplaceDb("DB.xls");
        // List list = new ArrayList();
        // list.add(new Integer(7566));
        // list.add(new Integer(7654));
        // なんですと？
        // List result = dao.getEmpTableByIds(list);
        List result = dao.getEmpTableByIds(7566, 7654);
        assertNotNull("1", result);
        assertEquals("2", 2, result.size());
    }

    public void testGetgetEmpTableWithRelation() {
        readXlsAllReplaceDb("DB.xls");
        EmpTableRelation result = dao.getEmpTableWithRelation(7369);
        assertNotNull(result);

    }

    public void testInsertTx() throws Exception {
        readXlsAllReplaceDb("DB.xls");
        EmpTable dto = createEntity();
        dao.insert(dto);
        assertEquals(15, readDbByTable("EMP_TABLE").getRowSize());
    }

    private EmpTable createEntity() {
        EmpTable dto = new EmpTable();
        dto.setEname("せとあずさ");
        dto.setJob("MANAGER");
        dto.setMgr(new Integer(7902));
        Calendar cal = Calendar.getInstance();
        cal.set(2006, 10, 1);
        dto.setHireDate(cal.getTime());
        dto.setSal(new BigDecimal(10000.0f));
        dto.setComm(new BigDecimal(1000.0f));
        dto.setDeptNo(new Integer(10));
        return dto;
    }

    public void testUpdateTx() throws Exception {

        readXlsAllReplaceDb("DB.xls");
        EmpTable dto = dao.getEmpTable(7369);
        dto.setEname(null);
        dao.update(dto);
        assertTable("EmpDaoTest_testUpdateTx.xls", "emp_table");
    }

    public void testUpdateNonStrictTx() throws Exception {

        readXlsAllReplaceDb("DB.xls");
        EmpTable dto = dao.getEmpTable(7369);
        dto.setEname(null);
        dto.setVersionno(10);
        {
            int result = dao.updateNonCheck(dto);
            assertEquals(0, result);
        }
        {
            int result = dao.updateNonCheckModifiedOnly(dto);
            assertEquals(0, result);
        }

    }

    public void testDeleteTx() throws Exception {
        readXlsAllReplaceDb("DB.xls");
        EmpTable dto = new EmpTable();
        dto.setEmpNo(7369);
        dto.setVersionno(1);
        dao.delete(dto);
        assertTable("EmpDaoTest_testDeleteTx.xls", "emp_table");
    }

    public void testFindTx() throws Exception {
        readXlsAllReplaceDb("DB.xls");
        EmpTableCondition dto = new EmpTableCondition();
        dto.selectEmpNo().more(7500);
        dto.selectHireDate().dateFrom(new Date(0, 0, 1));
        List<EmpTable> result = dao.find(dto);
        assertNotNull("1", result);
        assertEquals("2", 12, result.size());
    }

    public void testQueryTx() throws Exception {
        readXlsAllReplaceDb("DB.xls");
        EmpTableCondition dto = new EmpTableCondition();
        dto.selectEmpNo().not(7500);
        dto.addQuery("DEPT_NO <> ?", 10);
        dao.find(dto);
    }

    public void testQueryOnlyTx() throws Exception {
        readXlsAllReplaceDb("DB.xls");
        EmpTableCondition dto = new EmpTableCondition();
        dto.addQuery("DEPT_NO <> ?", 10);
        dao.find(dto);
    }

    public void testOptionAll() throws Exception {
        readXlsAllReplaceDb("DB.xls");
        EmpTableCondition condition = new EmpTableCondition();
        condition.selectComm().equal(new BigDecimal(1000));
        condition.selectDeptNo().in(10);
        {
            List<Integer> list = new ArrayList<Integer>();
            list.add(100);
            condition.selectEmpNo().in(list);
        }
        condition.selectEname().isNull();
        condition.selectHireDate().isNotNull();
        condition.selectMgr().less(1000);
        condition.selectSal().lessThan(new BigDecimal(100));
        condition.selectMgr().more(0);
        condition.selectSal().moreThan(BigDecimal.ZERO);
        condition.selectTstamp().not(new Timestamp(System.currentTimeMillis()));
        condition.selectVersionno().notIn(1);
        {
            List<Integer> list = new ArrayList<Integer>();
            list.add(99);
            condition.selectEmpNo().notIn(list);
        }

        dao.find(condition);

    }

    public void testSoleMatchTx() throws Exception {
        readXlsAllReplaceDb("DB.xls");
        EmpTableCondition dto = new EmpTableCondition();
        dto.selectEmpNo().moreThan(7934);
        dto.selectHireDate().more(new Date(0, 0, 1));
        System.err.println(dto);
        EmpTable result = dao.soleMatch(dto);
        assertNotNull("1", result);
    }

    public void testFindOrderByTx() throws Exception {
        EmpTableRelationCondition conditon = new EmpTableRelationCondition();
        conditon.selectEmpNo().addOrderAsc();
        conditon.selectDeptNo().addOrderDesc();
        System.err.println(conditon);
        dao.find(conditon);
    }

    public void testFindWithRelationTx() throws Exception {
        EmpTableRelationCondition condition = new EmpTableRelationCondition();
        condition.selectDeptTableRelation().selectLoc().equal("DALLAS");
        dao.findWithRelation(condition);
    }

    public void testFindByPager() throws Exception {
        readXlsAllReplaceDb("DB.xls");
        EmpTableCondition dto = new EmpTableCondition();
        dto.setOffset(5);
        dto.setLimit(5);
        dto.selectHireDate().more(new Date(0, 0, 1));
        dto.selectEmpNo().addOrderAsc();
        System.err.println(dto);

        List result = dao.find(dto);
        assertEquals(5, result.size());
        assertEquals(new Integer(7698), ((EmpTable) result.get(0)).getEmpNo());
        assertBeanListEquals("1", readXls("testFindByPager.xls"), result);
    }

    public void testUpdateModifiedOnly() throws Exception {
        readXlsAllReplaceDb("DB.xls");
        EmpTable dto = dao.getEmpTable(7654);
        dto.setEname("せとあずさ");
        dao.updateModifiedOnly(dto);
    }

    public void testInsertBatchTx() throws Exception {
        readXlsAllReplaceDb("DB.xls");
        List<EmpTable> list = new ArrayList<EmpTable>();
        {
            EmpTable dto = createEntity();
            dto.setEmpNo(9990);
            list.add(dto);
        }
        {
            EmpTable dto = createEntity();
            dto.setEmpNo(9991);
            list.add(dto);
        }
        dao.insertBatch(list);
        assertEquals(16, readDbByTable("EMP_TABLE").getRowSize());

    }

    public void testUpdateBatchTx() {
        readXlsAllReplaceDb("DB.xls");
        List<EmpTable> list = new ArrayList<EmpTable>();
        {
            EmpTable dto = createEntity();
            dto.setEmpNo(7369);
            list.add(dto);
        }
        {
            EmpTable dto = createEntity();
            dto.setEmpNo(7499);
            list.add(dto);
        }
        dao.updateBatch(list);
    }

    public void testUpdateDeleteTx() {
        readXlsAllReplaceDb("DB.xls");
        List<EmpTable> list = new ArrayList<EmpTable>();
        {
            EmpTable dto = createEntity();
            dto.setEmpNo(7369);
            list.add(dto);
        }
        {
            EmpTable dto = createEntity();
            dto.setEmpNo(7499);
            list.add(dto);
        }
        dao.deleteBatch(list);
        assertEquals(14, readDbByTable("EMP_TABLE").getRowSize());

    }

    // public void testObjectTx() throws Exception {
    // readXlsAllReplaceDb("DB.xls");
    // Object result = dao.getObject();
    // }
    //
    // public void testObject2Tx() throws Exception {
    // readXlsAllReplaceDb("DB.xls");
    // List<OriginalDto> result = dao.getObject2();
    // }

    protected String readText(String path) {
        InputStream is = ResourceUtil.getResourceAsStream(path);
        Reader reader = InputStreamReaderUtil.create(is, "JISAutoDetect");
        return ReaderUtil.readText(reader);
    }

    public void testPack() {
        readXlsAllReplaceDb("DB.xls");

        {
            List<EmpTableRelation> ret = dao.findWithRelation(new EmpTableRelationCondition());
            assertNotNull(ret.get(0).getDeptTableEntity());
        }
        {
            EmpTableRelation result = dao.getEmpTableWithRelation(7369);
            assertNotNull(result.getDeptTableEntity());
        }
        {
            List<EmpTableRelation> result = dao.getEmpTableByIdsWithRelation(7566, 7654);
            assertTrue(result.size() != 0);
            assertNotNull(result.get(0).getDeptTableEntity());
        }
        {
            List<EmpTableRelation> result = dao.getEmpTableByIdsWithRelation(7566, 7654);
            assertTrue(result.size() != 0);
            assertNotNull(result.get(0).getDeptTableEntity());
        }
        {
            EmpTableRelationCondition empTablePackFindDto = new EmpTableRelationCondition();
            empTablePackFindDto.selectEmpNo().equal(7654);
            EmpTableRelation result = dao.soleMatchWithRelation(empTablePackFindDto);
            assertNotNull(result.getDeptTableEntity());
        }

        {
            List<EmpTableRelation> result = dao.getAllEmpTableWithRelation();
            assertTrue(result.size() != 0);
            assertNotNull(result.get(0).getDeptTableEntity());

        }
    }

    public void testselectByUtf8SqlFile() {
        List<EmpTable> result = dao.selectByUtf8SqlFile();
        assertFalse(result.size() != 0);

    }

    /**
     * [DAO-65]
     */
    public void testFetch() {
        readXlsAllReplaceDb("DB.xls");
        final List<Boolean> azzert = new ArrayList<Boolean>();
        dao.select(new FetchHandler<EmpTable>() {

            public boolean execute(EmpTable bean) {
                azzert.add(Boolean.TRUE);
                return false;
            }
        });
        assertEquals(1, azzert.size());

    }
}
