/*
 * Copyright 2006-2011 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.codegen.lib;

import java.util.List;

import org.seasar.codegen.lib.exception.NullConditionValueException;

/**
 * 検索条件の要素を示すインターフェースです。
 * 
 * @author azusa
 * 
 * @param <T>
 */
public interface ConditionElement<T> {

    /**
     * 昇順の並び替えを指定します。
     */
    void addOrderAsc();

    /**
     * 降順の並び替えを指定します。
     */
    void addOrderDesc();

    /**
     * 値が等しい条件を示します。
     * 
     * @param value
     *            検索条件
     * @throws NullConditionValueException
     *             引数が<code>null</code>の場合
     */
    void equal(T value);

    /**
     * SQLのin句の条件を指定します。
     * 
     * @param values
     *            検索条件
     * @throws NullConditionValueException
     *             引数が<code>null</code>の場合
     */
    void in(T... values);

    /**
     * SQLのin句の条件を指定します
     * 
     * @param values
     *            検索条件
     * @throws NullConditionValueException
     *             引数が<code>null</code>の場合
     */
    void in(List<T> values);

    /**
     * SQLのNOT NULLの条件を指定します。
     */
    void isNotNull();

    /**
     * SQLのIS NULLの条件を指定します。
     */
    void isNull();

    /**
     * 未満の条件を指定します。
     * 
     * @param value
     *            検索条件
     * @throws NullConditionValueException
     *             引数が<code>null</code>の場合
     */
    void less(T value);

    /**
     * 以下の条件を指定します。
     * 
     * @param value
     *            検索条件
     * @throws NullConditionValueException
     *             引数が<code>null</code>の場合
     */
    void lessThan(T value);

    /**
     * 「より大きい」条件を指定します。
     * 
     * @param value
     *            検索条件
     * @throws NullConditionValueException
     *             引数が <code>null</code>の場合
     */
    void more(T value);

    /**
     * 以上の条件を指定します。
     * 
     * @param value
     *            検索条件
     * @throws NullConditionValueException
     *             引数が<code>null</code>の場合
     */
    void moreThan(T value);

    /**
     * 値が等しくない条件を指定します。
     * 
     * @param value
     *            検索条件
     * @throws NullConditionValueException
     *             引数が<code>null</code>の場合
     */
    void not(T value);

    /**
     * SQLのNOT IN句の条件を指定します。
     * 
     * @param values
     *            検索条件
     * @throws NullConditionValueException
     *             引数が<code>null</code>の場合
     */
    void notIn(T... values);

    /**
     * SQLのNOT IN句の条件を指定します。
     * 
     * @param values
     *            検索条件
     * @throws NullConditionValueException
     *             引数が<code>null</code>の場合
     */
    void notIn(List<T> values);

}