/*
 * Copyright 2004-2006 H2 Group. Licensed under the H2 License, Version 1.0 (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package org.h2.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;

public class IOUtils {
    
    private static final int BUFFER_BLOCK_SIZE = 4 * 1024;
    
    public static void closeSilently(OutputStream out) {
        if(out != null) {
            try {
                out.close();
            } catch(IOException e) {
                // ignore
            }
        }
    }

    public static void closeSilently(InputStream in) {
        if(in != null) {
            try {
                in.close();
            } catch(IOException e) {
                // ignore
            }
        }
    }
    
    public static void closeSilently(Reader reader) {
        if(reader != null) {
            try {
                reader.close();
            } catch(IOException e) {
                // ignore
            }
        }
    }

    public static void closeSilently(Writer writer) {
        if(writer != null) {
            try {
                writer.flush();
                writer.close();
            } catch(IOException e) {
                // ignore
            }
        }
    }

    public static byte[] readBytesAndClose(InputStream in, int length) throws IOException {
        try {
            if(length <= 0) {
                length = Integer.MAX_VALUE;
            }
            int block = Math.min(BUFFER_BLOCK_SIZE, length);
            ByteArrayOutputStream out=new ByteArrayOutputStream(block);
            byte[] buff=new byte[block];
            while(length > 0) {
                int len = Math.min(block, length);
                len = in.read(buff, 0, len);
                if(len < 0) {
                    break;
                }
                out.write(buff, 0, len);
                length -= len;
            }
            return out.toByteArray();
        } finally {
            in.close();
        }
    }
    
    public static String readStringAndClose(Reader in, int length) throws IOException {
        if(length <= 0) {
            length = Integer.MAX_VALUE;
        }
        int block = Math.min(BUFFER_BLOCK_SIZE, length);
        StringWriter out=new StringWriter(length == Integer.MAX_VALUE ? block : length);
        char[] buff=new char[block];
        while(length > 0) {
            int len = Math.min(block, length);
            len = in.read(buff, 0, len);
            if(len < 0) {
                break;
            }
            out.write(buff, 0, len);
            length -= len;
        }
        in.close();
        return out.toString();
    }

    public static int readFully(InputStream in, byte[] buffer, int max) throws IOException {
        int off = 0, len = Math.min(max, buffer.length);
        while (true) {
            int l = len - off;
            if (l <= 0) {
                break;
            }
            l = in.read(buffer, off, l);
            if (l == -1) {
                break;
            }
            off += l;
        }
        return off;
    }

    public static int readFully(Reader in, char[] buffer, int max) throws IOException {
        int off = 0, len = Math.min(max, buffer.length);
        while (true) {
            int l = len - off;
            if (l <= 0) {
                break;
            }
            l = in.read(buffer, off, l);
            if (l == -1) {
                break;
            }
            off += l;
        }
        return off;
    }
}
