/*
 * Copyright 2004-2006 H2 Group. Licensed under the H2 License, Version 1.0 (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package org.h2.expression;

import java.sql.SQLException;

import org.h2.command.dml.Query;
import org.h2.engine.Constants;
import org.h2.engine.Session;
import org.h2.message.Message;
import org.h2.result.LocalResult;
import org.h2.table.ColumnResolver;
import org.h2.table.TableFilter;
import org.h2.util.ObjectArray;
import org.h2.value.Value;
import org.h2.value.ValueArray;
import org.h2.value.ValueNull;

/**
 * @author Thomas
 */

public class Subquery extends Expression {

    private Query query;
    private long lastEvaluated;
    private Value lastResult;
    private Value[] lastParameters;

    public Subquery(Query query) {
        this.query = query;
    }

    public Value getValue(Session session) throws SQLException {
        query.setSession(session);
        ObjectArray list = query.getParameters();
        if(list == null) {
            list = new ObjectArray();
        }
        long now = session.getDatabase().getModificationDataId();
        // TODO refactoring: subquery and viewindex uses a similar (but not the same) mechanism to cache results
        if(Constants.OPTIMIZE_SUBQUERY_CACHE && query.isEverything(ExpressionVisitor.DETERMINISTIC) && query.isEverything(ExpressionVisitor.INDEPENDENT)) {
            boolean sameAsLast = (lastParameters != null);
            Value[] params = new Value[list.size()];
            for(int i=0; i<list.size(); i++) {
                Value v = (Value) list.get(i);
                params[i] = v;
                if(sameAsLast && !session.getDatabase().areEqual(lastParameters[i], v)) {
                    sameAsLast = false;
                }
            }
            if(query.getMaxDataModificationId() > lastEvaluated) {
                sameAsLast = false;
            }
            if(sameAsLast && lastResult != null) {
                return lastResult;
            }
            lastResult = null;
            lastParameters = params;
        }
        query.setSession(session);
        LocalResult result = query.query(2);
        this.lastEvaluated = now;
        try {
            int rowcount = result.getRowCount();
            if(rowcount > 1) {
                throw Message.getSQLException(Message.SCALAR_SUBQUERY_CONTAINS_MORE_THAN_ONE_ROW);
            }
            Value v;
            if(rowcount <= 0) {
                v = ValueNull.INSTANCE;
            } else {
                result.next();
                Value[] values = result.currentRow();
                if(result.getVisibleColumnCount() == 1) {
                    v = values[0];
                } else {
                    v = ValueArray.get(values);
                }
            }
            lastResult = v;
            return v;
        } finally {
            result.close();
        }
    }

    public int getType() {
        return getExpression().getType();
    }

    public void mapColumns(ColumnResolver resolver, int level) throws SQLException {
        query.mapColumns(resolver, level + 1);
    }

    public Expression optimize(Session session) throws SQLException {
        query.prepare();
        // TODO optimization: sometimes it is possible to optimize a subquery
        // can not evaluate, the data in the table(s) may change in some situations
        // for example if the subquery is used in a default value for a table
        //        ObjectArray params = query.getParameters();
        //        if(query.isConstant() || ((params == null || params.size()==0) && query.isDeterministic())) {
        //            return ValueExpression.get(getValue(session));
        //        }
        return this;
    }

    public void setEvaluatable(TableFilter tableFilter, boolean b) {
        query.setEvaluatable(tableFilter, b);
    }

    public int getScale() {
        return getExpression().getScale();
    }

    public long getPrecision() {
        return getExpression().getPrecision();
    }

    public String getSQL() {
        return "(" + query.getPlan() +")";
    }

    public void updateAggregate(Session session) {
        // TODO exists: is it possible that the subquery contains aggregates? probably not
    }

    private Expression getExpression() {
        return (Expression) query.getExpressions().get(0);
    }

    public boolean isEverything(ExpressionVisitor visitor) {
        return query.isEverything(visitor);
    }

    public Query getQuery() {
        return query;
    }

}
