/*
 * Copyright 2006-2011 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.codegen.impl;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.InputStream;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.seasar.codegen.element.Field;
import org.seasar.codegen.element.LinkTable;
import org.seasar.codegen.element.PrimaryKey;
import org.seasar.codegen.element.Table;
import org.seasar.codegen.util.ExcelUtil;
import org.seasar.framework.util.FileInputStreamUtil;
import org.seasar.framework.util.InputStreamUtil;
import org.seasar.framework.util.StringUtil;

/**
 * 1シート1テーブルの Excel ファイルを読み込みます。
 * 
 * @author glad
 */
public class ExcelImportCodeDataMultiSheet extends ExcelImportCodeDataSupport {

    protected int entityNameRow = 0;

    protected int entityNameCol = 1;

    protected int tableNameRow = 1;

    protected int tableNameCol = 1;

    protected int columnStartRow = 4;

    public ExcelImportCodeDataMultiSheet() {
        entityNameRow = 0;
        entityNameCol = 1;
        tableNameRow = 1;
        tableNameCol = 1;

        columnStartRow = 4;
        attributeNameCol = 0;
        columnNameCol = 1;
        dataTypeCol = 2;
        notNullCol = 3;
        primaryKeyCol = 4;
        foreignKeyCol = 5;
        defaultValueCol = 6;

        relationTypeCol = 7;
        parentTableCol = 8;
        parentColumnCol = 9;
        cardinalityCol = 10;
        parentPropertyCol = 11;
        childPropertyCol = 12;
    }

    public Map<String, Table> readCodeData(File srcFile) {
        InputStream in = FileInputStreamUtil.create(srcFile);
        try {
            in = new BufferedInputStream(in);
            HSSFWorkbook book = ExcelUtil.createWorkbook(in);
            Map<String, Table> tableMap = readCodeData(book);
            return tableMap;
        } finally {
            InputStreamUtil.close(in);
        }
    }

    protected Map<String, Table> readCodeData(HSSFWorkbook book) {
        Map<String, Table> tableMap = new LinkedHashMap<String, Table>();
        for (int i = 0; i < book.getNumberOfSheets(); ++i) {
            HSSFSheet sheet = book.getSheetAt(i);
            Table table = getTable(sheet);
            if (table != null) {
                tableMap.put(table.getTableName(), table);
            }
        }
        setupChildLinks(tableMap);
        return tableMap;
    }

    protected Table getTable(HSSFSheet sheet) {
        String tableName = getString(sheet, tableNameRow, tableNameCol);
        if (StringUtil.isEmpty(tableName)) {
            return null;
        }
        Table table = new Table();
        table.setTableName(tableName);

        for (int i = columnStartRow; i <= sheet.getLastRowNum(); ++i) {
            HSSFRow row = sheet.getRow(i);
            Field field = getField(row);
            if (field != null) {
                table.addTableField(field);

                PrimaryKey primaryKey = getPrimaryKey(row, field);
                if (primaryKey != null) {
                    table.addPrimaryKey(primaryKey);
                }
                LinkTable parentLink = getParentLink(row, field);
                if (parentLink != null) {
                    String relName = converter.convertParent(table, parentLink);
                    table.addLinkTable(relName, parentLink);
                }
            }
        }
        return table;
    }
}
