/*
 * Copyright 2006-2011 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.codegen.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.jiemamy.Jiemamy;
import org.jiemamy.ReferenceResolver;
import org.jiemamy.dialect.BuiltinDataTypeMold;
import org.jiemamy.model.RootModel;
import org.jiemamy.model.attribute.ColumnModel;
import org.jiemamy.model.attribute.ColumnRef;
import org.jiemamy.model.attribute.constraint.ForeignKey;
import org.jiemamy.model.datatype.BuiltinDataType;
import org.jiemamy.model.datatype.adapter.PrecisionedDataTypeAdapter;
import org.jiemamy.model.datatype.adapter.SerialDataTypeAdapter;
import org.jiemamy.model.datatype.adapter.SizedDataTypeAdapter;
import org.jiemamy.model.entity.EntityModel;
import org.jiemamy.model.entity.TableModel;
import org.jiemamy.serializer.JiemamySerializer;
import org.jiemamy.serializer.SerializationException;
import org.jiemamy.utils.EntityDependencyCalculator;
import org.seasar.codegen.CodeGenConfig;
import org.seasar.codegen.ImportCodeData;
import org.seasar.codegen.convert.FKNameConverter;
import org.seasar.codegen.dbms.Dbms;
import org.seasar.codegen.element.DataType;
import org.seasar.codegen.element.Field;
import org.seasar.codegen.element.FieldSetting;
import org.seasar.codegen.element.LinkTable;
import org.seasar.codegen.element.PrimaryKey;
import org.seasar.codegen.element.Table;
import org.seasar.codegen.exception.CodeGenException;
import org.seasar.codegen.exception.InternalGenerateException;
import org.seasar.codegen.exception.NotTypeMatchException;
import org.seasar.codegen.util.LinkUtil;
import org.seasar.codegen.util.SequnceUtil;
import org.seasar.framework.container.annotation.tiger.Binding;
import org.seasar.framework.container.annotation.tiger.BindingType;
import org.seasar.framework.util.StringUtil;

/**
 * <a href="http://www.jiemamy.org/">Jiemamy</a>の作成したER図を元にコードを生成するクラスです。
 * 
 * @author azusa
 * 
 */
public class JiemamyImportCodeData implements ImportCodeData {

    private Dbms dbms;

    private FKNameConverter converter;

    @Binding(bindingType = BindingType.MUST)
    private CodeGenConfig codeGenConfig;

    /*
     * (非 Javadoc)
     * 
     * @see org.seasar.codegen.ImportCodeData#readCodeData(java.io.File)
     */
    public Map<String, Table> readCodeData(File srcFile) {
        Jiemamy jiemamy = Jiemamy.newInstance();
        JiemamySerializer serializer = jiemamy.getSerializer();
        RootModel rootModel = null;
        try {
            rootModel = serializer.deserialize(new FileInputStream(srcFile));
        } catch (SerializationException e) {
            throw new CodeGenException("ECDG0004", e);
        } catch (FileNotFoundException e) {
            throw new CodeGenException("ECDG0004", e);
        }
        Map<String, Table> tableMap = new LinkedHashMap<String, Table>();
        for (EntityModel entityModel : EntityDependencyCalculator
                .getSortedEntityList(rootModel)) {

            if (entityModel instanceof TableModel) {
                TableModel tableModel = (TableModel) entityModel;
                Table table = getTable(tableModel, rootModel);
                List<LinkTable> parentLink = getParentLink(tableModel,
                        rootModel, table);
                for (LinkTable linkTable : parentLink) {
                    String relName = converter.convertParent(table, linkTable);
                    table.addLinkTable(relName, linkTable);
                }
                tableMap.put(table.getTableName(), table);
            }
        }

        setupChildLinks(tableMap);
        return tableMap;
    }

    Table getTable(TableModel tableModel, RootModel rootModel) {
        String tableName = tableModel.getName();
        Table table = new Table();
        table.setTableName(tableName);
        List<ColumnModel> columnModels = tableModel.findColumns();
        for (ColumnModel columnModel : columnModels) {
            Field field = getField(columnModel, rootModel);
            table.addTableField(field);
            PrimaryKey primaryKey = getPrimaryKey(columnModel, field);
            if (primaryKey != null) {
                table.addPrimaryKey(primaryKey);
            }
        }
        return table;
    }

    Field getField(ColumnModel columnModel, RootModel rootModel) {
        Field field = new Field();
        field.setFieldName(columnModel.getName());
        BuiltinDataType builtinDataType = getBuiltinDataType(columnModel);
        BuiltinDataTypeMold mold = null;
        try {
            mold = columnModel.getJiemamy().getDialect(rootModel)
                    .getMoldManager().findDataTypeMold(builtinDataType);
        } catch (ClassNotFoundException e) {
            throw new CodeGenException("ECDG0004", e);
        }
        FieldSetting fieldSetting = new FieldSetting();
        String langType = dbms.convDBTypeToDataType(mold.getName());
        fieldSetting.setTypeName(langType);
        int length = 0;
        if (builtinDataType.hasAdapter(SizedDataTypeAdapter.class)) {
            Integer ilength = builtinDataType.getAdapter(
                    SizedDataTypeAdapter.class).getSize();
            if (ilength != null) {
                length = ilength;
            }
        }
        int scale = 0;
        if (builtinDataType.hasAdapter(PrecisionedDataTypeAdapter.class)) {
            length = builtinDataType.getAdapter(
                    PrecisionedDataTypeAdapter.class).getPrecision();
            Integer iScale = builtinDataType.getAdapter(
                    PrecisionedDataTypeAdapter.class).getScale();
            if (iScale != null) {
                scale = iScale;
            }
        }
        fieldSetting.setColmnSize(length);
        fieldSetting.setPointNumber(scale);
        fieldSetting.setNotNull(columnModel.getNotNullConstraint() != null);
        if (columnModel.checkPrimaryKeyColumn()) {
            fieldSetting.setNotNull(true);
        }
        fieldSetting.setFieldDefault(columnModel.getDefaultValue() == null ? ""
                : columnModel.getDefaultValue());

        DataType dataType = null;
        try {
            dataType = dbms.selectBestDataType(fieldSetting);
        } catch (InternalGenerateException e) {
            throw new NotTypeMatchException(langType, langType, e);
        }
        field.setDataType(dataType);
        if (StringUtil.isEmpty(columnModel.getLogicalName())) {
            field.setFieldAttributeName(columnModel.getName());
        } else {
            field.setFieldAttributeName(columnModel.getLogicalName());
        }

        return field;
    }

    private BuiltinDataType getBuiltinDataType(ColumnModel columnModel) {
        ReferenceResolver referenceResolver = columnModel.getJiemamy()
                .getReferenceResolver();
        return columnModel.getDataType().toBuiltinDataType(referenceResolver);
    }

    PrimaryKey getPrimaryKey(ColumnModel columnModel, Field field) {
        if (columnModel.checkPrimaryKeyColumn()) {
            PrimaryKey primaryKey = new PrimaryKey();
            primaryKey.setField(field);
            if (getBuiltinDataType(columnModel).hasAdapter(
                    SerialDataTypeAdapter.class)) {
                field.setUseIdentity(true);
            }
            Map<String, String> sequenceMap = codeGenConfig.getSequnceMapping();
            if (sequenceMap != null) {
                String sequenceName = sequenceMap.get(columnModel
                        .findDeclaringTable().getName());
                SequnceUtil.addSequence(primaryKey, sequenceName);

            }

            return primaryKey;
        }
        return null;
    }

    List<LinkTable> getParentLink(TableModel tableModel, RootModel rootModel,
            Table table) {

        List<ForeignKey> foreignKeys = tableModel.findForeignKeys();
        List<LinkTable> linkTables = new ArrayList<LinkTable>(
                foreignKeys.size());

        for (ForeignKey foreignKey : foreignKeys) {
            LinkTable linkTable = new LinkTable();
            linkTable.setChildFieldName(findColumn(tableModel,
                    foreignKey.getKeyColumns(), linkTable));
            TableModel parentTable = (TableModel) foreignKey
                    .findReferencedEntity();
            linkTable.setParentFieldName(findColumn(parentTable,
                    foreignKey.getReferenceColumns(), linkTable));
            linkTable.setTableName(parentTable.getName());
            linkTables.add(linkTable);
        }
        return linkTables;
    }

    private String findColumn(TableModel tableModel,
            List<ColumnRef> columnRefs, LinkTable linkTable) {
        for (ColumnRef columnRef : columnRefs) {
            for (ColumnModel columnModel : tableModel.findColumns()) {
                if (columnModel.getId().equals(columnRef.getReferenceId())) {
                    return columnModel.getName();
                }
            }
        }
        throw new UnsupportedOperationException();
    }

    private void setupChildLinks(Map<String, Table> tableMap) {
        LinkUtil.setupChildLinks(tableMap);
    }

    public void setFKNameConverter(FKNameConverter converter) {
        this.converter = converter;
    }

    public void setDbms(Dbms dbms) {
        this.dbms = dbms;
    }

    public void setCodeGenConfig(CodeGenConfig codeGenConfig) {
        this.codeGenConfig = codeGenConfig;
    }

}
