/*
 * Copyright 2004-2014 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.extension.jdbc.types;

import java.sql.CallableStatement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;

import org.seasar.extension.jdbc.util.BindVariableUtil;

/**
 * EnumをJDBCで扱うためのクラスです。
 * 
 * @author higa
 * 
 */
public class EnumType extends AbstractValueType {

    @SuppressWarnings("unchecked")
    private final Class<? extends Enum> enumClass;

    /**
     * <code>EnumType</code>を返します。
     * 
     * @param enumClass
     */
    @SuppressWarnings("unchecked")
    public EnumType(Class<? extends Enum> enumClass) {
        super(Types.VARCHAR);
        this.enumClass = enumClass;
    }

    public Object getValue(ResultSet resultSet, int index) throws SQLException {
        return toEnum(resultSet.getString(index));
    }

    /**
     * {@link Enum}に変換します。
     * 
     * @param name
     * @return {@link Enum}
     */
    @SuppressWarnings("unchecked")
    protected Enum toEnum(String name) {
        if (name == null) {
            return null;
        }
        return Enum.valueOf(enumClass, name);
    }

    public Object getValue(ResultSet resultSet, String columnName)
            throws SQLException {

        return toEnum(resultSet.getString(columnName));
    }

    public Object getValue(CallableStatement cs, int index) throws SQLException {
        return toEnum(cs.getString(index));
    }

    public Object getValue(CallableStatement cs, String parameterName)
            throws SQLException {

        return toEnum(cs.getString(parameterName));
    }

    @SuppressWarnings("unchecked")
    public void bindValue(PreparedStatement ps, int index, Object value)
            throws SQLException {

        if (value == null) {
            setNull(ps, index);
        } else {
            ps.setString(index, (Enum.class.cast(value)).name());
        }
    }

    @SuppressWarnings("unchecked")
    public void bindValue(CallableStatement cs, String parameterName,
            Object value) throws SQLException {

        if (value == null) {
            setNull(cs, parameterName);
        } else {
            cs.setString(parameterName, (Enum.class.cast(value)).name());
        }
    }

    public String toText(Object value) {
        if (value == null) {
            return BindVariableUtil.nullText();
        }
        return BindVariableUtil.toText((Enum.class.cast(value)).name());
    }

}